#!/bin/bash
# CC-ADVANCECYBER Installer
# One-command installation of Claude Code enhancements

set -euo pipefail

VERSION="1.0.0"
SCRIPT_DIR="$(cd "$(dirname "${BASH_SOURCE[0]}")" && pwd)"
CLAUDE_DIR="$HOME/.claude"
DOCKER_DIR="$HOME/docker/local"
PROFILE="full"
FORCE=false
SKIP_BACKUP=false
DRY_RUN=false
TARGET_DIR=""

# Colors
RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
BLUE='\033[0;34m'
NC='\033[0m'

# Logging
log_info() { echo -e "${GREEN}[INFO]${NC} $1"; }
log_warn() { echo -e "${YELLOW}[WARN]${NC} $1"; }
log_error() { echo -e "${RED}[ERROR]${NC} $1"; }
log_step() { echo -e "${BLUE}[STEP]${NC} $1"; }

# Usage
usage() {
    cat << EOF
CC-ADVANCECYBER Installer v${VERSION}

Usage: ./install.sh [OPTIONS]

Options:
    --profile <full|lite>    Installation profile (default: full)
    --target-dir <path>      Install to custom directory (isolated testing)
    --dry-run                Show what would be done without making changes
    --force                  Overwrite without confirmation
    --skip-backup            Skip backup creation
    -h, --help              Show this help message

Profiles:
    full    All components including Docker memory backend
    lite    Agents and skills only (no Docker dependencies)

Examples:
    ./install.sh                              # Full installation to ~/.claude
    ./install.sh --profile lite               # Lite installation
    ./install.sh --target-dir /tmp/test-cc    # Isolated test installation
    ./install.sh --dry-run                    # Preview without installing
EOF
    exit 0
}

# Parse arguments
parse_args() {
    while [[ $# -gt 0 ]]; do
        case $1 in
            --profile)
                PROFILE="$2"
                shift 2
                ;;
            --force)
                FORCE=true
                shift
                ;;
            --skip-backup)
                SKIP_BACKUP=true
                shift
                ;;
            --target-dir)
                TARGET_DIR="$2"
                shift 2
                ;;
            --dry-run)
                DRY_RUN=true
                shift
                ;;
            -h|--help)
                usage
                ;;
            *)
                log_error "Unknown option: $1"
                usage
                ;;
        esac
    done

    if [[ "$PROFILE" != "full" && "$PROFILE" != "lite" ]]; then
        log_error "Invalid profile: $PROFILE"
        exit 1
    fi

    # Apply target directory if specified
    if [[ -n "$TARGET_DIR" ]]; then
        CLAUDE_DIR="$TARGET_DIR/.claude"
        DOCKER_DIR="$TARGET_DIR/docker/local"
        log_info "Using isolated target: $TARGET_DIR"
    fi
}

# Prerequisite checks
check_prerequisites() {
    log_step "Checking prerequisites..."

    # Check if ~/.claude exists (Claude Code installed) - skip for isolated testing
    if [ ! -d "$CLAUDE_DIR" ]; then
        if [[ -n "$TARGET_DIR" ]]; then
            log_info "Creating isolated test directory: $CLAUDE_DIR"
            mkdir -p "$CLAUDE_DIR"
        else
            log_error "Claude Code not found. Please install Claude Code first."
            log_info "Visit: https://claude.ai/download"
            exit 1
        fi
    fi

    # Full profile checks
    if [ "$PROFILE" = "full" ]; then
        # Check Docker
        if ! command -v docker &> /dev/null; then
            log_error "Docker is required for full installation"
            log_info "Install Docker Desktop: https://www.docker.com/products/docker-desktop"
            exit 1
        fi

        if ! docker info &> /dev/null; then
            log_error "Docker is not running"
            log_info "Please start Docker Desktop"
            exit 1
        fi

        # Check Ollama (optional but recommended)
        if ! command -v ollama &> /dev/null; then
            log_warn "Ollama not found. Memory embeddings will require Ollama."
            log_info "Install Ollama: https://ollama.ai"
        fi
    fi

    log_info "Prerequisites check passed"
}

# Create backup
create_backup() {
    if [ "$SKIP_BACKUP" = true ]; then
        log_warn "Skipping backup (--skip-backup)"
        return
    fi

    log_step "Creating backup..."

    local TIMESTAMP=$(date +%Y%m%d-%H%M%S)

    # Backup ~/.claude
    if [ -d "$CLAUDE_DIR/agents" ] || [ -d "$CLAUDE_DIR/skills" ]; then
        local BACKUP_DIR="$CLAUDE_DIR/backups/install-$TIMESTAMP"
        mkdir -p "$BACKUP_DIR"

        if [ -d "$CLAUDE_DIR/agents" ]; then
            cp -r "$CLAUDE_DIR/agents" "$BACKUP_DIR/"
        fi
        if [ -d "$CLAUDE_DIR/skills" ]; then
            cp -r "$CLAUDE_DIR/skills" "$BACKUP_DIR/"
        fi
        if [ -f "$CLAUDE_DIR/CLAUDE.md" ]; then
            cp "$CLAUDE_DIR/CLAUDE.md" "$BACKUP_DIR/"
        fi

        log_info "Backed up to $BACKUP_DIR"
    fi
}

# Install agents
install_agents() {
    log_step "Installing agents..."

    mkdir -p "$CLAUDE_DIR/agents"

    local COUNT=0
    for agent in "$SCRIPT_DIR/agents"/*.md; do
        if [ -f "$agent" ]; then
            cp "$agent" "$CLAUDE_DIR/agents/"
            COUNT=$((COUNT + 1))
        fi
    done

    # Copy manifest
    if [ -f "$SCRIPT_DIR/agents/index.json" ]; then
        cp "$SCRIPT_DIR/agents/index.json" "$CLAUDE_DIR/agents/"
    fi

    log_info "Installed $COUNT agents"
}

# Install skills
install_skills() {
    log_step "Installing skills..."

    mkdir -p "$CLAUDE_DIR/skills"

    local COUNT=0
    for skill in "$SCRIPT_DIR/skills"/*/; do
        if [ -d "$skill" ]; then
            local skillname=$(basename "$skill")
            cp -r "$skill" "$CLAUDE_DIR/skills/$skillname"
            COUNT=$((COUNT + 1))
        fi
    done

    # Copy manifest
    if [ -f "$SCRIPT_DIR/skills/index.json" ]; then
        cp "$SCRIPT_DIR/skills/index.json" "$CLAUDE_DIR/skills/"
    fi

    log_info "Installed $COUNT skills"
}

# Install CLAUDE.md template
install_claude_md() {
    log_step "Configuring CLAUDE.md..."

    local TEMPLATE="$SCRIPT_DIR/config/CLAUDE.md.template"
    local TARGET="$CLAUDE_DIR/CLAUDE.md"

    if [ -f "$TARGET" ] && [ "$FORCE" != true ]; then
        # Merge: append template to existing
        log_info "Merging with existing CLAUDE.md"
        echo "" >> "$TARGET"
        echo "# --- CC-ADVANCECYBER Additions ---" >> "$TARGET"
        cat "$TEMPLATE" >> "$TARGET"
    else
        cp "$TEMPLATE" "$TARGET"
    fi

    log_info "CLAUDE.md configured"
}

# Install MCP configuration (full profile)
install_mcp_config() {
    if [ "$PROFILE" != "full" ]; then
        return
    fi

    log_step "Configuring MCP servers..."

    local TEMPLATE="$SCRIPT_DIR/config/settings.local.json.template"
    local TARGET="$CLAUDE_DIR/settings.local.json"

    if [ -f "$TARGET" ]; then
        log_warn "settings.local.json exists - manual merge may be required"
        log_info "Template available at: $TEMPLATE"
    else
        # Replace placeholders with environment variable syntax
        cp "$TEMPLATE" "$TARGET"
    fi
}

# Setup Docker memory backend (full profile)
setup_memory_backend() {
    if [ "$PROFILE" != "full" ]; then
        log_info "Skipping memory backend (lite profile)"
        return
    fi

    log_step "Setting up memory backend..."

    mkdir -p "$DOCKER_DIR"

    # Copy Docker files
    cp "$SCRIPT_DIR/memory/docker-compose.yml" "$DOCKER_DIR/"
    cp "$SCRIPT_DIR/memory/setup.sh" "$DOCKER_DIR/"
    cp "$SCRIPT_DIR/memory/.env.template" "$DOCKER_DIR/.env.template"

    # Run setup
    chmod +x "$DOCKER_DIR/setup.sh"
    cd "$DOCKER_DIR"
    ./setup.sh

    log_info "Memory backend configured at $DOCKER_DIR"
}

# Install CLI command
install_cli() {
    log_step "Installing advancecyber CLI..."

    local BIN_DIR="$HOME/.local/bin"
    local CLI_SOURCE="$SCRIPT_DIR/bin/cc-advancecyber"
    local CLI_TARGET="$BIN_DIR/advancecyber"

    # Create bin directory
    mkdir -p "$BIN_DIR"

    # Copy CLI
    cp "$CLI_SOURCE" "$CLI_TARGET"
    chmod +x "$CLI_TARGET"

    # Check if ~/.local/bin is in PATH
    if [[ ":$PATH:" != *":$BIN_DIR:"* ]]; then
        log_warn "~/.local/bin is not in PATH"
        log_info "Add to your shell profile: export PATH=\"\$HOME/.local/bin:\$PATH\""
    fi

    log_info "CLI installed: $CLI_TARGET"
}

# Verify installation
verify_installation() {
    log_step "Verifying installation..."

    local ERRORS=0

    # Check agent count
    local AGENT_COUNT=$(ls -1 "$CLAUDE_DIR/agents"/*.md 2>/dev/null | wc -l | tr -d ' ')
    if [ "$AGENT_COUNT" -lt 40 ]; then
        log_warn "Expected 46 agents, found $AGENT_COUNT"
        ERRORS=$((ERRORS + 1))
    else
        log_info "Agents: $AGENT_COUNT installed"
    fi

    # Check skill count
    local SKILL_COUNT=$(find "$CLAUDE_DIR/skills" -mindepth 1 -maxdepth 1 -type d 2>/dev/null | wc -l | tr -d ' ')
    if [ "$SKILL_COUNT" -lt 30 ]; then
        log_warn "Expected 40 skills, found $SKILL_COUNT"
        ERRORS=$((ERRORS + 1))
    else
        log_info "Skills: $SKILL_COUNT installed"
    fi

    # Check memory backend (full profile)
    if [ "$PROFILE" = "full" ]; then
        if docker compose -f "$DOCKER_DIR/docker-compose.yml" ps 2>/dev/null | grep -q "healthy"; then
            log_info "Memory backend: healthy"
        else
            log_warn "Memory backend: may still be starting"
        fi
    fi

    if [ $ERRORS -eq 0 ]; then
        log_info "Verification passed"
    else
        log_warn "Verification completed with $ERRORS warnings"
    fi
}

# Print completion message
print_completion() {
    echo ""
    echo "=============================================="
    echo -e "${GREEN}  CC-ADVANCECYBER Installation Complete!${NC}"
    echo "=============================================="
    echo ""
    echo "Profile: $PROFILE"
    echo ""
    echo "Launch with:"
    echo ""
    echo "    advancecyber"
    echo ""

    if [ "$PROFILE" = "full" ]; then
        echo "To start memory backend:"
        echo "    advancecyber memory start"
        echo ""
    fi

    echo "Documentation: https://advancecyber.ai/docs"
    echo ""
}

# Main
main() {
    echo ""
    echo "=============================================="
    echo "  CC-ADVANCECYBER Installer v${VERSION}"
    echo "=============================================="
    echo ""

    parse_args "$@"

    log_info "Profile: $PROFILE"
    echo ""

    check_prerequisites
    create_backup
    install_agents
    install_skills
    install_claude_md
    install_mcp_config
    install_cli
    setup_memory_backend
    verify_installation
    print_completion
}

main "$@"
