#!/bin/bash
# CC-ADVANCECYBER Memory Backend Setup

set -euo pipefail

SCRIPT_DIR="$(cd "$(dirname "${BASH_SOURCE[0]}")" && pwd)"
ENV_FILE="$SCRIPT_DIR/.env"

echo "CC-ADVANCECYBER Memory Backend Setup"
echo "======================================"

# Check Docker
if ! command -v docker &> /dev/null; then
    echo "ERROR: Docker is not installed"
    echo "Please install Docker Desktop: https://www.docker.com/products/docker-desktop"
    exit 1
fi

# Check Docker Compose
if ! docker compose version &> /dev/null; then
    echo "ERROR: Docker Compose is not available"
    exit 1
fi

# Generate .env if not exists
if [ ! -f "$ENV_FILE" ]; then
    echo "Generating secure credentials..."

    QDRANT_KEY=$(openssl rand -hex 32)
    POSTGRES_PASS=$(openssl rand -base64 24 | tr -d '/+=' | head -c 32)
    N8N_PASS=$(openssl rand -base64 16 | tr -d '/+=' | head -c 16)

    cat > "$ENV_FILE" << EOF
# CC-ADVANCECYBER Memory Backend Configuration
# Generated: $(date -u +%Y-%m-%dT%H:%M:%SZ)

QDRANT_API_KEY=$QDRANT_KEY
POSTGRES_PASSWORD=$POSTGRES_PASS
N8N_USER=admin
N8N_PASSWORD=$N8N_PASS
TZ=America/New_York
EOF

    chmod 600 "$ENV_FILE"
    echo "Credentials generated and saved to .env"
    echo ""
    echo "IMPORTANT: Store these credentials securely!"
    echo "  Qdrant API Key: $QDRANT_KEY"
    echo "  PostgreSQL Password: $POSTGRES_PASS"
    echo "  n8n Password: $N8N_PASS"
else
    echo "Using existing .env file"
fi

# Start services
echo ""
echo "Starting memory backend..."
cd "$SCRIPT_DIR"
docker compose up -d

# Wait for health
echo "Waiting for services to be healthy..."
sleep 10

# Health check
echo ""
echo "Checking service health..."

if docker compose ps | grep -q "healthy"; then
    echo "All services are healthy!"
else
    echo "WARNING: Some services may still be starting"
    docker compose ps
fi

echo ""
echo "Memory backend is ready!"
echo "  Qdrant:     http://localhost:6334"
echo "  PostgreSQL: localhost:5432"
echo "  n8n:        http://localhost:5679"
