#!/bin/bash
# Check distribution package for secrets and PII

set -euo pipefail

DIST_DIR="${1:-.}"

echo "Sanitization Check"
echo "=================="
echo "Directory: $DIST_DIR"
echo ""

FAILED=0

check_pattern() {
    local name="$1"
    local pattern="$2"

    if grep -rq "$pattern" "$DIST_DIR" 2>/dev/null; then
        echo "FAIL: $name detected"
        grep -rn "$pattern" "$DIST_DIR" 2>/dev/null | head -3
        FAILED=1
    else
        echo "OK: No $name"
    fi
}

echo "Checking for secrets..."
echo "-----------------------"

# API keys
check_pattern "Anthropic API key" "sk-ant-"
check_pattern "OpenRouter API key" "sk-or-"
check_pattern "Generic API key pattern" 'sk-[a-zA-Z0-9]\{20,\}'

# Credentials
check_pattern "Password in config" 'password.*=.*[^$]'
check_pattern "API key in config" 'api.key.*=.*[^$]'

echo ""
echo "Checking for PII..."
echo "-------------------"

# Email addresses
if grep -rqE "[a-zA-Z0-9._%+-]+@gmail\.com" "$DIST_DIR" 2>/dev/null; then
    echo "FAIL: Gmail address detected"
    FAILED=1
else
    echo "OK: No Gmail addresses"
fi

# SSH connections
check_pattern "SSH connection" 'root@[0-9]'

# Known personal patterns
check_pattern "Travel preferences" 'flying out of BOS\|Diamond (Hilton)'

echo ""
echo "Checking for private agents..."
echo "------------------------------"

PRIVATE_AGENTS=("advisor" "convert" "equity-research-analyst" "intel" "ma-research" "skill-updater" "travel" "uthink")

for agent in "${PRIVATE_AGENTS[@]}"; do
    if [ -f "$DIST_DIR/agents/${agent}.md" ]; then
        echo "FAIL: Private agent found: ${agent}.md"
        FAILED=1
    else
        echo "OK: ${agent}.md not present"
    fi
done

echo ""
echo "=================="

if [ $FAILED -eq 0 ]; then
    echo "PASS: Sanitization check passed"
    exit 0
else
    echo "FAIL: Sanitization issues detected"
    echo ""
    echo "DO NOT DISTRIBUTE until issues are resolved"
    exit 1
fi
