#!/bin/bash
# Verify agent file integrity against manifest checksums

set -euo pipefail

AGENTS_DIR="${1:-$HOME/.claude/agents}"
MANIFEST="$AGENTS_DIR/index.json"

if [ ! -f "$MANIFEST" ]; then
    echo "ERROR: Manifest not found: $MANIFEST"
    exit 1
fi

if ! command -v jq &> /dev/null; then
    echo "ERROR: jq is required but not installed"
    exit 1
fi

echo "Verifying agent integrity..."
echo "Manifest: $MANIFEST"
echo ""

PASSED=0
FAILED=0
MISSING=0

while IFS= read -r agent; do
    checksum=$(echo "$agent" | jq -r '.sha256')
    file=$(echo "$agent" | jq -r '.file')
    name=$(echo "$agent" | jq -r '.name')

    if [ ! -f "$AGENTS_DIR/$file" ]; then
        echo "MISSING: $file"
        MISSING=$((MISSING + 1))
        continue
    fi

    actual=$(shasum -a 256 "$AGENTS_DIR/$file" | cut -d' ' -f1)

    if [ "$checksum" = "$actual" ]; then
        echo "OK: $name"
        PASSED=$((PASSED + 1))
    else
        echo "MISMATCH: $name"
        echo "  Expected: $checksum"
        echo "  Actual:   $actual"
        FAILED=$((FAILED + 1))
    fi
done < <(jq -c '.agents[]' "$MANIFEST")

echo ""
echo "================================"
echo "Verification Summary:"
echo "  Passed:  $PASSED"
echo "  Failed:  $FAILED"
echo "  Missing: $MISSING"
echo "================================"

if [ $FAILED -gt 0 ] || [ $MISSING -gt 0 ]; then
    echo ""
    echo "INTEGRITY CHECK FAILED"
    exit 1
else
    echo ""
    echo "All agents verified successfully"
    exit 0
fi
