#!/bin/bash
# Post-installation verification

set -euo pipefail

CLAUDE_DIR="${1:-$HOME/.claude}"
DOCKER_DIR="${2:-$HOME/docker/local}"
PROFILE="${3:-full}"

echo "CC-ADVANCECYBER Installation Verification"
echo "=========================================="
echo "Profile: $PROFILE"
echo ""

ERRORS=0
WARNINGS=0

# Check function
check() {
    local name="$1"
    local condition="$2"
    local is_error="${3:-true}"

    if eval "$condition"; then
        echo "[OK] $name"
    else
        if [ "$is_error" = true ]; then
            echo "[FAIL] $name"
            ERRORS=$((ERRORS + 1))
        else
            echo "[WARN] $name"
            WARNINGS=$((WARNINGS + 1))
        fi
    fi
}

echo "Core Components:"
echo "----------------"
check "Claude directory exists" "[ -d '$CLAUDE_DIR' ]"
check "Agents directory exists" "[ -d '$CLAUDE_DIR/agents' ]"
check "Skills directory exists" "[ -d '$CLAUDE_DIR/skills' ]"
check "CLAUDE.md exists" "[ -f '$CLAUDE_DIR/CLAUDE.md' ]"

# Agent count
AGENT_COUNT=$(ls -1 "$CLAUDE_DIR/agents"/*.md 2>/dev/null | wc -l | tr -d ' ')
check "Agent count >= 40" "[ $AGENT_COUNT -ge 40 ]"

# Skill count
SKILL_COUNT=$(find "$CLAUDE_DIR/skills" -mindepth 1 -maxdepth 1 -type d 2>/dev/null | wc -l | tr -d ' ')
check "Skill count >= 30" "[ $SKILL_COUNT -ge 30 ]"

# Manifest files
check "Agent manifest exists" "[ -f '$CLAUDE_DIR/agents/index.json' ]"
check "Skill manifest exists" "[ -f '$CLAUDE_DIR/skills/index.json' ]" false

# Full profile checks
if [ "$PROFILE" = "full" ]; then
    echo ""
    echo "Full Profile Components:"
    echo "------------------------"
    check "Docker directory exists" "[ -d '$DOCKER_DIR' ]"
    check "docker-compose.yml exists" "[ -f '$DOCKER_DIR/docker-compose.yml' ]"
    check ".env file exists" "[ -f '$DOCKER_DIR/.env' ]"
    check "settings.local.json exists" "[ -f '$CLAUDE_DIR/settings.local.json' ]" false

    # Docker services
    if command -v docker &> /dev/null && docker info &> /dev/null; then
        check "Docker is running" "true"

        # Check containers
        if [ -f "$DOCKER_DIR/docker-compose.yml" ]; then
            cd "$DOCKER_DIR"
            check "Qdrant container exists" "docker compose ps | grep -q qdrant" false
            check "PostgreSQL container exists" "docker compose ps | grep -q postgres" false
            check "n8n container exists" "docker compose ps | grep -q n8n" false
        fi
    else
        check "Docker is running" "false"
    fi

    # Ollama
    if command -v ollama &> /dev/null; then
        check "Ollama installed" "true"
        check "Ollama responding" "ollama list &>/dev/null" false
    else
        check "Ollama installed" "false" false
    fi
fi

echo ""
echo "=========================================="
echo "Summary:"
echo "  Errors:   $ERRORS"
echo "  Warnings: $WARNINGS"
echo "=========================================="

if [ $ERRORS -gt 0 ]; then
    echo ""
    echo "VERIFICATION FAILED"
    exit 1
else
    echo ""
    echo "Verification passed"
    if [ $WARNINGS -gt 0 ]; then
        echo "(with $WARNINGS warnings)"
    fi
    exit 0
fi
