---
name: hostinger
description: Manage Hostinger services via API including domains, VPS, hosting, DNS, billing, and email marketing. Use when the user needs to automate Hostinger infrastructure, manage domains/DNS, control VPS instances, deploy Docker containers, manage billing/subscriptions, or work with email marketing contacts and segments.
---

# Hostinger API Integration

Access and manage Hostinger services programmatically via their REST API. Supports domain management, VPS control, DNS configuration, hosting services, billing operations, and email marketing automation.

## API Fundamentals

### Base Configuration

- **Base URL**: `https://developers.hostinger.com`
- **OpenAPI Spec**: `https://developers.hostinger.com/openapi/openapi.json`
- **Content-Type**: `application/json` (all requests)
- **Authentication**: Bearer token

### Authentication

```bash
Authorization: Bearer YOUR_API_TOKEN
```

Tokens are created via the Hostinger Panel account page. Tokens inherit the permissions of the user who created them.

### Pagination

- Default: 50 items per page
- Use `page` query parameter for pagination
- Response metadata includes: `current_page`, `per_page`, `total`

### Rate Limiting

- Rate limits are enforced per endpoint
- 429 responses indicate limit exceeded
- Repeated violations result in IP blocking
- Domain availability check: 10 requests/minute

### Error Handling

```json
{
  "error": "Error message description",
  "correlation_id": "unique-request-id"
}
```

422 responses include field-level errors:
```json
{
  "message": "Validation failed",
  "errors": {
    "field_name": ["Error detail"]
  },
  "correlation_id": "unique-request-id"
}
```

---

## API Categories

### 1. Domain Management

#### Check Domain Availability
```
GET /api/domains/v1/availability?domain=example.com
```
Rate limit: 10 requests/minute

#### Domain Portfolio
- `GET /api/domains/v1/portfolio` - List owned domains
- `POST /api/domains/v1/portfolio` - Purchase domain
- `GET /api/domains/v1/portfolio/{domain}` - Get domain details
- `PUT /api/domains/v1/portfolio/{domain}/lock` - Toggle domain lock
- `PUT /api/domains/v1/portfolio/{domain}/privacy` - Toggle WHOIS privacy
- `PUT /api/domains/v1/portfolio/{domain}/nameservers` - Update nameservers

#### Domain Forwarding
- `GET /api/domains/v1/forwarding/{domain}` - Get forwarding config
- `PUT /api/domains/v1/forwarding/{domain}` - Set redirect (301/302)
- `DELETE /api/domains/v1/forwarding/{domain}` - Remove forwarding

#### WHOIS Profiles
- `GET /api/domains/v1/whois` - List contact profiles
- `POST /api/domains/v1/whois` - Create contact profile
- `PUT /api/domains/v1/whois/{id}` - Update profile
- `DELETE /api/domains/v1/whois/{id}` - Delete profile

---

### 2. DNS Management

#### DNS Zones
- `GET /api/dns/v1/zones/{domain}` - Get DNS records
- `PUT /api/dns/v1/zones/{domain}` - Update DNS records
- `DELETE /api/dns/v1/zones/{domain}/{record_id}` - Delete record
- `POST /api/dns/v1/zones/{domain}/validate` - Validate before changes

#### DNS Snapshots (Backups)
- `GET /api/dns/v1/snapshots/{domain}` - List snapshots
- `POST /api/dns/v1/snapshots/{domain}` - Create snapshot
- `POST /api/dns/v1/snapshots/{domain}/{snapshot_id}/restore` - Restore

---

### 3. VPS Management

#### Virtual Machines
- `GET /api/vps/v1/virtual-machines` - List VPS instances
- `POST /api/vps/v1/virtual-machines` - Purchase/setup VPS
- `GET /api/vps/v1/virtual-machines/{id}` - Get VPS details
- `POST /api/vps/v1/virtual-machines/{id}/start` - Start instance
- `POST /api/vps/v1/virtual-machines/{id}/stop` - Stop instance
- `POST /api/vps/v1/virtual-machines/{id}/restart` - Restart instance
- `POST /api/vps/v1/virtual-machines/{id}/recreate` - Recreate (deletes all data!)
- `POST /api/vps/v1/virtual-machines/{id}/recovery` - Boot recovery mode

#### VPS Firewall
- `GET /api/vps/v1/firewall/{vm_id}/rules` - List firewall rules
- `POST /api/vps/v1/firewall/{vm_id}/rules` - Create rule
- `DELETE /api/vps/v1/firewall/{vm_id}/rules/{rule_id}` - Delete rule
- `POST /api/vps/v1/firewall/{vm_id}/activate` - Activate firewall
- `POST /api/vps/v1/firewall/{vm_id}/deactivate` - Deactivate firewall
- `POST /api/vps/v1/firewall/{vm_id}/sync` - Sync rules to VM

Supported protocols: TCP, UDP, ICMP, GRE, custom
**Note**: Firewall drops all incoming traffic by default; explicit accept rules required.

#### VPS Docker Manager (Experimental)
- `GET /api/vps/v1/docker/{vm_id}/projects` - List projects
- `POST /api/vps/v1/docker/{vm_id}/projects` - Deploy from compose file
- `GET /api/vps/v1/docker/{vm_id}/projects/{name}` - Get project
- `DELETE /api/vps/v1/docker/{vm_id}/projects/{name}` - Delete project
- `GET /api/vps/v1/docker/{vm_id}/containers` - List containers
- `POST /api/vps/v1/docker/{vm_id}/containers/{id}/start` - Start container
- `POST /api/vps/v1/docker/{vm_id}/containers/{id}/stop` - Stop container
- `GET /api/vps/v1/docker/{vm_id}/containers/{id}/logs` - Get logs

Project names: alphanumeric with dashes/underscores, 3-64 characters.

#### VPS Snapshots
- `GET /api/vps/v1/snapshots/{vm_id}` - List snapshots
- `POST /api/vps/v1/snapshots/{vm_id}` - Create snapshot
- `POST /api/vps/v1/snapshots/{vm_id}/{snapshot_id}/restore` - Restore

#### VPS Backups
- `GET /api/vps/v1/backups/{vm_id}` - List weekly backups
- `POST /api/vps/v1/backups/{vm_id}/{backup_id}/restore` - Restore backup

#### SSH Public Keys
- `GET /api/vps/v1/public-keys` - List SSH keys
- `POST /api/vps/v1/public-keys` - Add SSH key
- `DELETE /api/vps/v1/public-keys/{id}` - Remove key
- `POST /api/vps/v1/virtual-machines/{id}/public-keys` - Attach key to VM

#### Post-Install Scripts
- `GET /api/vps/v1/post-install-scripts` - List scripts
- `POST /api/vps/v1/post-install-scripts` - Create script (max 48KB)
- `DELETE /api/vps/v1/post-install-scripts/{id}` - Delete script

#### PTR Records (Reverse DNS)
- `GET /api/vps/v1/ptr/{vm_id}` - Get PTR record
- `PUT /api/vps/v1/ptr/{vm_id}` - Set PTR record

#### OS Templates
- `GET /api/vps/v1/templates` - List available OS templates

#### Metrics
- `GET /api/vps/v1/metrics/{vm_id}` - Get performance metrics

Query params: `from`, `to` (date range), `metric` (cpu, ram, disk, network, uptime)

---

### 4. Hosting Services

#### Websites
- `GET /api/hosting/v1/websites` - List websites
- `POST /api/hosting/v1/websites` - Create website
- `GET /api/hosting/v1/websites/{id}` - Get website details
- `GET /api/hosting/v1/websites/{id}/domains` - List website domains
- `POST /api/hosting/v1/websites/{id}/domains` - Add domain to website

#### Datacenters
- `GET /api/hosting/v1/datacenters` - List available regions

#### Free Subdomains
- `POST /api/hosting/v1/subdomains` - Generate free subdomain

#### Domain Verification
- `GET /api/hosting/v1/verification/pending` - Pending verifications
- `GET /api/hosting/v1/verification/verified` - Completed verifications
- `POST /api/hosting/v1/verification/{domain}` - Verify via TXT record

---

### 5. Billing & Subscriptions

#### Catalog
- `GET /api/billing/v1/catalog` - Browse service plans

**Note**: Prices are in cents (e.g., 999 = $9.99)

#### Payment Methods
- `GET /api/billing/v1/payment-methods` - List payment methods
- `POST /api/billing/v1/payment-methods/{id}/default` - Set default

#### Subscriptions
- `GET /api/billing/v1/subscriptions` - List active subscriptions
- `GET /api/billing/v1/subscriptions/{id}` - Get subscription details
- `PUT /api/billing/v1/subscriptions/{id}/auto-renewal` - Toggle auto-renewal
- `POST /api/billing/v1/subscriptions/{id}/cancel` - Cancel subscription

#### Orders
- `POST /api/billing/v1/orders` - Create service order
- `GET /api/billing/v1/orders/{id}` - Get order status

**Note**: Domain and VPS orders partially deprecated; use dedicated endpoints.

---

### 6. Email Marketing (Reach)

#### Contacts
- `GET /api/reach/v1/contacts` - List contacts
- `POST /api/reach/v1/contacts` - Create contact
- `GET /api/reach/v1/contacts/{id}` - Get contact
- `PUT /api/reach/v1/contacts/{id}` - Update contact
- `DELETE /api/reach/v1/contacts/{id}` - Delete contact

#### Segments
- `GET /api/reach/v1/segments` - List segments
- `POST /api/reach/v1/segments` - Create segment
- `GET /api/reach/v1/segments/{id}` - Get segment
- `PUT /api/reach/v1/segments/{id}` - Update segment
- `DELETE /api/reach/v1/segments/{id}` - Delete segment

Segment conditions: email, name, subscription_status
Logic: AND/OR operators supported

---

## Code Examples

### TypeScript/JavaScript

```typescript
const HOSTINGER_API = 'https://developers.hostinger.com';
const API_TOKEN = process.env.HOSTINGER_API_TOKEN;

async function hostingerRequest<T>(
  endpoint: string,
  options: RequestInit = {}
): Promise<T> {
  const response = await fetch(`${HOSTINGER_API}${endpoint}`, {
    ...options,
    headers: {
      'Authorization': `Bearer ${API_TOKEN}`,
      'Content-Type': 'application/json',
      ...options.headers,
    },
  });

  if (!response.ok) {
    const error = await response.json();
    throw new Error(`Hostinger API error: ${error.error} (${error.correlation_id})`);
  }

  return response.json();
}

// Check domain availability
async function checkDomainAvailability(domain: string) {
  return hostingerRequest<{available: boolean; price?: number}>(
    `/api/domains/v1/availability?domain=${encodeURIComponent(domain)}`
  );
}

// List VPS instances
async function listVPS() {
  return hostingerRequest<{data: VPSInstance[]}>('/api/vps/v1/virtual-machines');
}

// Start VPS
async function startVPS(vmId: string) {
  return hostingerRequest(`/api/vps/v1/virtual-machines/${vmId}/start`, {
    method: 'POST'
  });
}

// Update DNS records
async function updateDNS(domain: string, records: DNSRecord[]) {
  return hostingerRequest(`/api/dns/v1/zones/${domain}`, {
    method: 'PUT',
    body: JSON.stringify({ records })
  });
}
```

### Python

```python
import os
import requests

HOSTINGER_API = 'https://developers.hostinger.com'
API_TOKEN = os.environ.get('HOSTINGER_API_TOKEN')

def hostinger_request(endpoint: str, method: str = 'GET', data: dict = None):
    headers = {
        'Authorization': f'Bearer {API_TOKEN}',
        'Content-Type': 'application/json'
    }

    url = f'{HOSTINGER_API}{endpoint}'
    response = requests.request(method, url, headers=headers, json=data)

    if not response.ok:
        error = response.json()
        raise Exception(f"Hostinger API error: {error.get('error')} ({error.get('correlation_id')})")

    return response.json()

# Check domain availability
def check_domain(domain: str):
    return hostinger_request(f'/api/domains/v1/availability?domain={domain}')

# List VPS instances
def list_vps():
    return hostinger_request('/api/vps/v1/virtual-machines')

# Create firewall rule
def add_firewall_rule(vm_id: str, rule: dict):
    return hostinger_request(
        f'/api/vps/v1/firewall/{vm_id}/rules',
        method='POST',
        data=rule
    )

# Deploy Docker project
def deploy_docker(vm_id: str, name: str, compose_content: str):
    return hostinger_request(
        f'/api/vps/v1/docker/{vm_id}/projects',
        method='POST',
        data={'name': name, 'compose': compose_content}
    )
```

### cURL Examples

```bash
# Check domain availability
curl -X GET "https://developers.hostinger.com/api/domains/v1/availability?domain=example.com" \
  -H "Authorization: Bearer $HOSTINGER_API_TOKEN"

# List VPS instances
curl -X GET "https://developers.hostinger.com/api/vps/v1/virtual-machines" \
  -H "Authorization: Bearer $HOSTINGER_API_TOKEN"

# Start VPS
curl -X POST "https://developers.hostinger.com/api/vps/v1/virtual-machines/{vm_id}/start" \
  -H "Authorization: Bearer $HOSTINGER_API_TOKEN"

# Update DNS
curl -X PUT "https://developers.hostinger.com/api/dns/v1/zones/example.com" \
  -H "Authorization: Bearer $HOSTINGER_API_TOKEN" \
  -H "Content-Type: application/json" \
  -d '{"records": [{"type": "A", "name": "@", "content": "1.2.3.4", "ttl": 3600}]}'

# Create firewall rule
curl -X POST "https://developers.hostinger.com/api/vps/v1/firewall/{vm_id}/rules" \
  -H "Authorization: Bearer $HOSTINGER_API_TOKEN" \
  -H "Content-Type: application/json" \
  -d '{"protocol": "TCP", "port": "443", "source": "0.0.0.0/0", "action": "accept"}'
```

---

## Important Constraints

### Passwords
- Minimum 12 characters
- Must include: uppercase, lowercase, numbers
- Cannot be publicly leaked credentials

### Docker Projects
- Names: alphanumeric, dashes, underscores only
- Length: 3-64 characters

### Nameserver Changes
- Risk of availability issues if misconfigured
- API provides warnings for improper configuration

### VPS Recreation
- **Irreversible**: Deletes all snapshots and data
- Use with extreme caution

### TLD Requirements
- Some TLDs require additional registration details beyond standard WHOIS

---

## Official SDKs & Integrations

| Platform | Type |
|----------|------|
| Ansible | Collection |
| PHP | SDK |
| Python | SDK |
| Node/TypeScript | SDK |
| Terraform | Provider |
| WHMCS | Module |
| CLI | Tool |
| n8n | Integration |
| MCP Server | AI Integration |
| Postman | Collection |

---

## Workflows

### Domain Registration Flow
1. Check availability: `GET /api/domains/v1/availability?domain=...`
2. Create/select WHOIS profile: `POST /api/domains/v1/whois`
3. Purchase domain: `POST /api/domains/v1/portfolio`
4. Configure DNS: `PUT /api/dns/v1/zones/{domain}`
5. Enable privacy: `PUT /api/domains/v1/portfolio/{domain}/privacy`

### VPS Setup Flow
1. List templates: `GET /api/vps/v1/templates`
2. Add SSH key: `POST /api/vps/v1/public-keys`
3. Create post-install script (optional): `POST /api/vps/v1/post-install-scripts`
4. Purchase/setup VPS: `POST /api/vps/v1/virtual-machines`
5. Configure firewall: `POST /api/vps/v1/firewall/{vm_id}/rules`
6. Activate firewall: `POST /api/vps/v1/firewall/{vm_id}/activate`

### Docker Deployment Flow
1. Prepare docker-compose.yml content
2. Deploy project: `POST /api/vps/v1/docker/{vm_id}/projects`
3. Check container status: `GET /api/vps/v1/docker/{vm_id}/containers`
4. View logs: `GET /api/vps/v1/docker/{vm_id}/containers/{id}/logs`

---

## References

- **API Documentation**: https://developers.hostinger.com/
- **OpenAPI Spec**: https://developers.hostinger.com/openapi/openapi.json
- **Postman Collection**: Import via Postman using collection ID `36145449-4a733c4f-6704-49f6-832a-0ccd28c37021`
