---
name: testing-security-stack
description: Comprehensive Docker-based testing and security scanning infrastructure for automated quality assurance, security testing, performance testing, and API validation. Use when writing tests, running security scans, performing load testing, executing API tests, generating test reports, or orchestrating multi-tool testing workflows. Covers SonarQube, Semgrep, Trivy, OWASP ZAP, Nuclei, Gitleaks, Newman, WireMock, K6, Locust, Artillery, Playwright, Allure, CAI, Checkov, DefectDojo, Syft, Pact, Pa11y, Toxiproxy, RESTler, BackstopJS, Falco, and Flyway.
---

# Testing & Security Stack

Complete guide to orchestrating comprehensive testing and security workflows using Docker-based tools for code quality, security scanning, performance testing, API testing, chaos engineering, and AI-powered security analysis.

## Stack Overview

The testing-security-stack provides 25+ integrated services across 11 categories:

| Category | Services | Purpose |
|----------|----------|---------|
| **Code Quality** | SonarQube, Semgrep | Static analysis, code quality |
| **Security Testing** | OWASP ZAP, Nuclei, Gitleaks, Checkov, Falco | Dynamic security, secret detection, IaC, runtime |
| **Container Security** | Trivy, Syft | Vulnerability scanning, SBOM generation |
| **AI Security** | CAI | AI-powered offensive/defensive security automation |
| **API Testing** | Newman, WireMock, Pact, RESTler | API testing, mocking, contracts, fuzzing |
| **Performance Testing** | K6, Locust, Artillery, Toxiproxy | Load testing, stress testing, chaos engineering |
| **Browser Testing** | Playwright, BackstopJS, Pa11y | E2E, visual regression, accessibility |
| **Database Testing** | Flyway | Migration testing, schema validation |
| **Vulnerability Management** | DefectDojo | Finding aggregation, tracking, reporting |
| **Reporting** | Allure | Test result aggregation and visualization |
| **Supply Chain** | Syft, Trivy | SBOM, dependency analysis |

## Quick Start Commands

```bash
# Location: /Users/marcvance/Documents/Code/testing-security-stack

# Start all services
make up

# Run all scans
make scan-all

# Run security scans only
make security-scan

# Run performance tests
make perf-test

# Run API tests
make api-test

# Stop services
make down
```

## Service Access Points

| Service | URL | Default Credentials |
|---------|-----|---------------------|
| SonarQube | http://localhost:9000 | admin/admin |
| OWASP ZAP | http://localhost:8082 | - |
| WireMock | http://localhost:8081 | - |
| Allure UI | http://localhost:5252 | - |
| Dashboard | http://localhost:10000 | - |
| Locust | http://localhost:8089 | - |
| Playwright | http://localhost:3000 | - |
| DefectDojo | http://localhost:8083 | admin/defectdojo |
| Pact Broker | http://localhost:9292 | - |
| Toxiproxy API | http://localhost:8474 | - |

## Test Orchestration Workflows

### Workflow 1: Full Security Scan Pipeline

**When**: Before code deployment or during security audit

**Steps**:
1. **Start services**:
   ```bash
   cd /Users/marcvance/Documents/Code/testing-security-stack
   make up
   ```

2. **Run static analysis (SAST)**:
   ```bash
   # Semgrep for security patterns
   docker exec semgrep semgrep --config auto --json --output /reports/results.json /src

   # Trivy for container vulnerabilities
   docker exec trivy trivy fs --format json -o /reports/trivy.json /src
   ```

3. **Run dynamic analysis (DAST)**:
   ```bash
   # OWASP ZAP baseline scan
   docker exec owasp-zap zap-baseline.py -t http://target:8080 -r /zap/reports/baseline.html

   # Nuclei vulnerability scan
   docker exec nuclei nuclei -target http://target:8080 -o /reports/nuclei.txt
   ```

4. **Check for secrets**:
   ```bash
   docker exec gitleaks gitleaks detect --source /src --report-path /reports/secrets.json
   ```

5. **View results in Allure**: http://localhost:5252

**Reference**: See [security-testing.md](references/security-testing.md) for detailed security scanning patterns.

### Workflow 2: Performance Testing Pipeline

**When**: Before release to validate load handling

**Steps**:
1. **Run smoke test** (quick sanity):
   ```bash
   docker exec k6 k6 run --vus 2 --duration 30s /scripts/smoke.js
   ```

2. **Run load test** (normal load):
   ```bash
   docker exec k6 k6 run /scripts/load-test.js
   ```

3. **Run stress test** (find limits):
   ```bash
   docker exec locust locust -f /locust/locustfile.py --headless -u 1000 -r 50 -t 5m
   ```

4. **Run spike test** (sudden surge):
   ```bash
   docker exec artillery artillery run /scripts/spike-test.yml
   ```

5. **Analyze results**:
   - K6 JSON output: `./reports/k6/`
   - Locust reports: `./reports/locust/`
   - Artillery reports: `./reports/artillery/`

**Reference**: See [performance-testing.md](references/performance-testing.md) for load testing patterns.

### Workflow 3: API Testing Pipeline

**When**: Testing API contracts and endpoints

**Steps**:
1. **Start mock services**:
   ```bash
   docker exec wiremock curl -X POST http://localhost:8080/__admin/mappings \
     -d '{"request":{"method":"GET","url":"/api/health"},"response":{"status":200}}'
   ```

2. **Run Newman tests**:
   ```bash
   docker exec newman newman run /etc/newman/collection.json \
     -e /etc/newman/environment.json \
     --reporters cli,json,allure \
     --reporter-json-export /results/newman.json
   ```

3. **Run contract tests**:
   ```bash
   docker exec newman newman run /etc/newman/contract-tests.json
   ```

4. **Security test APIs**:
   ```bash
   docker exec owasp-zap zap-api-scan.py -t http://wiremock:8080/api -f openapi
   ```

**Reference**: See [api-testing.md](references/api-testing.md) for API testing patterns.

### Workflow 4: E2E Browser Testing

**When**: Validating user workflows

**Steps**:
1. **Run Playwright tests**:
   ```bash
   docker exec playwright npx playwright test --reporter=allure-playwright
   ```

2. **Run visual regression tests**:
   ```bash
   docker exec playwright npx playwright test --grep @visual
   ```

3. **Run security-focused E2E tests**:
   ```bash
   docker exec playwright npx playwright test --grep @security
   ```

4. **Generate report**:
   ```bash
   docker exec playwright npx playwright show-report
   ```

**Reference**: See [browser-testing.md](references/browser-testing.md) for Playwright patterns.

### Workflow 5: AI-Powered Security Analysis

**When**: Deep security analysis with AI agents

**Steps**:
1. **Start CAI container**:
   ```bash
   docker exec -it cai /opt/cai/start.sh
   ```

2. **Use specialized agents**:
   ```bash
   # Bug bounty hunting
   CAI>/agent bug_bounter_agent
   CAI> Scan https://target.com for vulnerabilities

   # Red team assessment
   CAI>/agent redteam_agent
   CAI> Enumerate attack surface for target application

   # Forensic analysis
   CAI>/agent dfir_agent
   CAI> Analyze suspicious log entries in /logs
   ```

3. **Generate reports**:
   ```bash
   CAI>/agent reporting_agent
   CAI> Generate security assessment report for findings
   ```

**Reference**: See [ai-security.md](references/ai-security.md) for CAI agent patterns.

### Workflow 6: Infrastructure as Code Security

**When**: Validating Terraform, Kubernetes, or Dockerfile security

**Steps**:
1. **Scan Terraform configurations**:
   ```bash
   docker exec checkov checkov -d /tf --framework terraform --output sarif --output-file /reports/checkov.sarif
   ```

2. **Scan Kubernetes manifests**:
   ```bash
   docker exec checkov checkov -d /k8s --framework kubernetes --output json --output-file /reports/k8s-scan.json
   ```

3. **Scan Dockerfiles**:
   ```bash
   docker exec checkov checkov -f /src/Dockerfile --framework dockerfile
   ```

4. **Generate compliance report**:
   ```bash
   docker exec checkov checkov -d /iac --check CIS --output cli
   ```

**Reference**: See [checkov.md](references/checkov.md) for IaC security patterns.

### Workflow 7: Contract Testing Pipeline

**When**: Validating API contracts between services

**Steps**:
1. **Run consumer contract tests**:
   ```bash
   docker exec pact npm test -- --testPathPattern=pact
   ```

2. **Publish contracts to broker**:
   ```bash
   docker exec pact pact-broker publish ./pacts --consumer-app-version $(git rev-parse HEAD) --broker-base-url http://pact-broker:9292
   ```

3. **Verify provider**:
   ```bash
   docker exec pact npx jest --testPathPattern=pact-provider
   ```

4. **Check deployment safety**:
   ```bash
   docker exec pact pact-broker can-i-deploy --pacticipant consumer --version $(git rev-parse HEAD) --to production
   ```

**Reference**: See [pact.md](references/pact.md) for contract testing patterns.

### Workflow 8: Chaos Engineering Pipeline

**When**: Testing resilience to failures

**Steps**:
1. **Create proxy for service**:
   ```bash
   docker exec toxiproxy toxiproxy-cli create --listen 0.0.0.0:25000 --upstream backend:8080 backend-proxy
   ```

2. **Add latency toxic**:
   ```bash
   docker exec toxiproxy toxiproxy-cli toxic add --type latency --attribute latency=2000 --attribute jitter=500 backend-proxy
   ```

3. **Run resilience tests**:
   ```bash
   docker exec k6 k6 run /scripts/resilience-test.js
   ```

4. **Test with connection failures**:
   ```bash
   docker exec toxiproxy toxiproxy-cli toxic add --type reset_peer --attribute timeout=0 backend-proxy
   ```

**Reference**: See [toxiproxy.md](references/toxiproxy.md) for chaos engineering patterns.

### Workflow 9: Accessibility Testing Pipeline

**When**: Validating WCAG compliance

**Steps**:
1. **Run accessibility scan**:
   ```bash
   docker exec pa11y pa11y http://target:8080 --standard WCAG2AA --reporter json > /reports/a11y.json
   ```

2. **Run CI configuration**:
   ```bash
   docker exec pa11y pa11y-ci --config /config/.pa11yci.json
   ```

3. **Generate HTML report**:
   ```bash
   docker exec pa11y pa11y http://target:8080 --reporter html > /reports/accessibility.html
   ```

**Reference**: See [pa11y.md](references/pa11y.md) for accessibility testing patterns.

### Workflow 10: Visual Regression Pipeline

**When**: Detecting unintended UI changes

**Steps**:
1. **Create reference screenshots**:
   ```bash
   docker exec backstopjs backstop reference
   ```

2. **Run visual comparison**:
   ```bash
   docker exec backstopjs backstop test
   ```

3. **Approve changes** (if intentional):
   ```bash
   docker exec backstopjs backstop approve
   ```

4. **View report**:
   ```bash
   open backstop_data/html_report/index.html
   ```

**Reference**: See [backstopjs.md](references/backstopjs.md) for visual regression patterns.

### Workflow 11: API Fuzzing Pipeline

**When**: Deep security testing of REST APIs

**Steps**:
1. **Compile OpenAPI spec**:
   ```bash
   docker exec restler restler compile --api_spec /specs/openapi.json
   ```

2. **Run quick fuzz test**:
   ```bash
   docker exec restler restler fuzz-lean --grammar_file /Compile/grammar.py --dictionary_file /Compile/dict.json
   ```

3. **Run full fuzzing**:
   ```bash
   docker exec restler restler fuzz --grammar_file /Compile/grammar.py --dictionary_file /Compile/dict.json --time_budget 1
   ```

4. **Analyze bug buckets**:
   ```bash
   find /results/Fuzz/bug_buckets -name "*.txt" -exec cat {} \;
   ```

**Reference**: See [restler.md](references/restler.md) for API fuzzing patterns.

### Workflow 12: Database Migration Testing

**When**: Validating schema migrations

**Steps**:
1. **Show migration status**:
   ```bash
   docker exec flyway flyway info
   ```

2. **Run migrations**:
   ```bash
   docker exec flyway flyway migrate
   ```

3. **Validate migrations**:
   ```bash
   docker exec flyway flyway validate
   ```

4. **Repair if needed**:
   ```bash
   docker exec flyway flyway repair
   ```

**Reference**: See [flyway.md](references/flyway.md) for database migration patterns.

### Workflow 13: Supply Chain Security

**When**: Generating SBOM and tracking dependencies

**Steps**:
1. **Generate SBOM**:
   ```bash
   docker exec syft syft /src -o spdx-json=/reports/sbom.spdx.json
   ```

2. **Scan SBOM for vulnerabilities**:
   ```bash
   docker exec grype grype sbom:/reports/sbom.spdx.json --output sarif > /reports/vulnerabilities.sarif
   ```

3. **Generate CycloneDX format**:
   ```bash
   docker exec syft syft /src -o cyclonedx-json=/reports/sbom.cdx.json
   ```

**Reference**: See [syft.md](references/syft.md) for SBOM generation patterns.

### Workflow 14: Vulnerability Management

**When**: Aggregating and tracking security findings

**Steps**:
1. **Import Semgrep results**:
   ```bash
   curl -X POST "http://localhost:8083/api/v2/import-scan/" \
     -H "Authorization: Token $DEFECTDOJO_TOKEN" \
     -F "scan_type=Semgrep JSON Report" \
     -F "file=@/reports/semgrep.json" \
     -F "engagement=1"
   ```

2. **Import Trivy results**:
   ```bash
   curl -X POST "http://localhost:8083/api/v2/import-scan/" \
     -H "Authorization: Token $DEFECTDOJO_TOKEN" \
     -F "scan_type=Trivy Scan" \
     -F "file=@/reports/trivy.json" \
     -F "engagement=1"
   ```

3. **View findings**: http://localhost:8083

**Reference**: See [defectdojo.md](references/defectdojo.md) for vulnerability management patterns.

### Workflow 15: Runtime Security Monitoring

**When**: Detecting threats during test execution

**Steps**:
1. **Start Falco monitoring**:
   ```bash
   docker-compose up -d falco
   ```

2. **Run E2E tests with monitoring**:
   ```bash
   docker exec playwright npx playwright test
   ```

3. **Check for security alerts**:
   ```bash
   docker logs falco 2>&1 | grep "Warning\|Error\|Critical"
   ```

**Reference**: See [falco.md](references/falco.md) for runtime security patterns.

## Tool Decision Trees

### Choosing Security Scanning Tools

```
Static Analysis (SAST):
├── Code quality + coverage → SonarQube
├── Security patterns + rules → Semgrep
├── Container/FS vulnerabilities → Trivy
├── Secret detection → Gitleaks
├── IaC security (Terraform/K8s) → Checkov
└── Supply chain/SBOM → Syft + Grype

Dynamic Analysis (DAST):
├── Web app scanning → OWASP ZAP
├── Template-based scanning → Nuclei
├── Stateful API fuzzing → RESTler
├── Runtime monitoring → Falco
└── AI-powered analysis → CAI
```

### Choosing API Testing Tools

```
API Testing:
├── Collection-based testing → Newman
├── Service mocking → WireMock
├── Contract testing → Pact
├── API fuzzing → RESTler
└── Security testing → OWASP ZAP API Scan
```

### Choosing Performance Testing Tools

```
Performance Testing:
├── Developer-friendly (JS) → K6
│   └── Best for: CI/CD integration, scripted tests
├── Distributed testing (Python) → Locust
│   └── Best for: Large scale, real-time dashboard
├── Scenario-based (YAML) → Artillery
│   └── Best for: Complex scenarios, protocol variety
└── Chaos engineering → Toxiproxy
    └── Best for: Failure injection, resilience testing
```

### Choosing Browser Testing Tools

```
Browser Testing:
├── E2E functional → Playwright
├── Visual regression → BackstopJS
├── Accessibility (WCAG) → Pa11y
└── Security-focused E2E → Playwright + custom tests
```

### Choosing Database Testing Tools

```
Database Testing:
├── Migration testing → Flyway
├── Schema validation → Flyway validate
└── Data integrity → Custom tests
```

### Choosing Test Type

```
Test Selection:
├── Smoke Test → Quick sanity check (1-2 VUs, 1-2min)
├── Load Test → Normal load validation (expected VUs, 15-60min)
├── Stress Test → Find breaking points (above normal, until failure)
├── Spike Test → Sudden surge handling (spike to high, short burst)
├── Soak Test → Endurance testing (normal load, hours)
├── Chaos Test → Failure injection (latency, timeouts, resets)
└── Contract Test → API compatibility verification
```

### Vulnerability Management Flow

```
Finding Management:
├── Aggregate findings → DefectDojo
├── Track remediation → DefectDojo engagements
├── Risk assessment → DefectDojo risk scoring
└── Report generation → DefectDojo reports
```

## Common Patterns & Solutions

### Pattern 1: CI/CD Security Gate

```yaml
# GitHub Actions example
security-scan:
  steps:
    - name: SAST Scan
      run: |
        docker run --rm -v ${PWD}:/src returntocorp/semgrep \
          semgrep --config auto --json -o results.json /src

    - name: Container Scan
      run: |
        docker run --rm -v ${PWD}:/src aquasec/trivy \
          trivy fs --severity HIGH,CRITICAL --exit-code 1 /src

    - name: Secret Scan
      run: |
        docker run --rm -v ${PWD}:/src zricethezav/gitleaks \
          detect --source /src --exit-code 1
```

### Pattern 2: Pre-Deployment Testing

```bash
#!/bin/bash
# run-tests.sh

# 1. Static Analysis
make security-scan
SAST_EXIT=$?

# 2. API Contract Tests
make api-test
API_EXIT=$?

# 3. E2E Tests
make e2e-test
E2E_EXIT=$?

# 4. Performance Baseline
make perf-smoke
PERF_EXIT=$?

# Check all passed
if [ $SAST_EXIT -eq 0 ] && [ $API_EXIT -eq 0 ] && [ $E2E_EXIT -eq 0 ] && [ $PERF_EXIT -eq 0 ]; then
  echo "All tests passed - ready for deployment"
  exit 0
else
  echo "Tests failed - blocking deployment"
  exit 1
fi
```

### Pattern 3: Result Aggregation

```bash
# Aggregate all results to Allure
docker exec allure allure generate /app/allure-results -o /app/allure-reports --clean

# Access dashboard
open http://localhost:5252
```

## Quality Gates & Thresholds

### Security Thresholds

| Severity | Action | Threshold |
|----------|--------|-----------|
| Critical | Block deployment | 0 findings |
| High | Block deployment | 0 findings |
| Medium | Review required | < 5 findings |
| Low | Track in backlog | Unlimited |

### Performance Thresholds

| Metric | Target | Fail Threshold |
|--------|--------|----------------|
| Response Time (p95) | < 200ms | > 500ms |
| Error Rate | 0% | > 1% |
| Throughput | > 1000 req/s | < 500 req/s |
| CPU Usage | < 70% | > 90% |

### Code Quality Thresholds

| Metric | Target | Fail Threshold |
|--------|--------|----------------|
| Code Coverage | > 80% | < 60% |
| Duplication | < 3% | > 5% |
| Maintainability | A | < B |
| Security Hotspots | 0 | > 0 |

## Troubleshooting Guide

### Services Not Starting

```bash
# Check service status
docker-compose ps

# View logs
docker-compose logs -f [service-name]

# Common fix: Reset volumes
docker-compose down -v
docker-compose up -d
```

### SonarQube Issues

**Problem**: Elasticsearch memory errors
```bash
# Fix: Increase vm.max_map_count
sudo sysctl -w vm.max_map_count=262144
```

**Problem**: Database connection errors
```bash
# Check PostgreSQL is running
docker-compose logs sonarqube-db
```

### ZAP Scan Failures

**Problem**: Target unreachable
```bash
# Ensure target is on same Docker network
docker network connect testing-network target-container
```

**Problem**: Scan too slow
```bash
# Use baseline scan for quick results
docker exec owasp-zap zap-baseline.py -t http://target:8080
```

### Performance Test Issues

**Problem**: K6 not generating output
```bash
# Check output configuration
docker exec k6 k6 run --out json=/reports/results.json /scripts/test.js
```

**Problem**: Locust not finding locustfile
```bash
# Ensure volume mount is correct
docker exec locust ls -la /locust/
```

## Best Practices Summary

### Security Testing
1. Run SAST before DAST (faster feedback)
2. Use quality gates to block deployments
3. Scan containers and dependencies
4. Check for secrets in every commit
5. Aggregate findings in single dashboard

### Performance Testing
1. Start with smoke tests (quick feedback)
2. Establish baselines before changes
3. Test with realistic data volumes
4. Include think time in scripts
5. Monitor resource usage during tests

### API Testing
1. Organize collections by feature
2. Use environments for different stages
3. Include negative test cases
4. Validate contracts not just status codes
5. Test authentication and authorization

### E2E Testing
1. Use Page Object Model pattern
2. Isolate test data
3. Run critical paths first
4. Include visual regression
5. Test accessibility

### Reporting
1. Centralize all results in Allure
2. Include environment information
3. Tag tests by category/priority
4. Preserve history for trends
5. Generate executive summaries

## Reference Documentation

### Core Testing Guides

#### [Security Testing Guide](references/security-testing.md)
- SAST with Semgrep and SonarQube
- DAST with OWASP ZAP
- Container scanning with Trivy
- Secret detection with Gitleaks
- Template-based scanning with Nuclei

#### [Performance Testing Guide](references/performance-testing.md)
- K6 load testing patterns
- Locust distributed testing
- Artillery scenario testing
- Threshold configuration
- Result analysis

#### [API Testing Guide](references/api-testing.md)
- Newman collection organization
- WireMock service virtualization
- Contract testing patterns
- API security testing
- CI/CD integration

#### [Browser Testing Guide](references/browser-testing.md)
- Playwright test organization
- Page Object Model
- Visual regression testing
- Security-focused E2E tests
- Parallel execution

#### [AI Security Guide](references/ai-security.md)
- CAI agent overview
- Red team operations
- Bug bounty workflows
- Digital forensics
- Report generation

#### [Code Quality Guide](references/code-quality.md)
- SonarQube configuration
- Quality gates
- Coverage requirements
- Technical debt management
- Integration patterns

### Extended Tool Guides

#### [IaC Security - Checkov](references/checkov.md)
- Terraform/Kubernetes scanning
- Custom check development
- Compliance frameworks (CIS, SOC2, HIPAA)
- CI/CD integration patterns

#### [Vulnerability Management - DefectDojo](references/defectdojo.md)
- Scan result import (Semgrep, Trivy, ZAP, etc.)
- Finding lifecycle management
- JIRA integration
- Reporting and metrics

#### [SBOM Generation - Syft](references/syft.md)
- SPDX/CycloneDX output formats
- Package catalogers
- Grype vulnerability scanning
- Attestation workflows

#### [Contract Testing - Pact](references/pact.md)
- Consumer-driven contracts
- Provider verification
- Pact Broker management
- Can-I-Deploy workflow

#### [Accessibility Testing - Pa11y](references/pa11y.md)
- WCAG 2.1 AA compliance
- Authenticated page testing
- CI/CD configuration
- Common issue remediation

#### [Chaos Engineering - Toxiproxy](references/toxiproxy.md)
- Proxy management
- Toxic types (latency, timeout, bandwidth)
- Resilience testing patterns
- Circuit breaker validation

#### [API Fuzzing - RESTler](references/restler.md)
- OpenAPI spec compilation
- Fuzz-lean quick security check
- Full stateful fuzzing
- Bug bucket analysis

#### [Visual Regression - BackstopJS](references/backstopjs.md)
- Viewport configuration
- Scenario interactions
- Custom scripts
- CI/CD integration

#### [Runtime Security - Falco](references/falco.md)
- Custom rule development
- Alert priorities
- Kubernetes integration
- Sidekick forwarding

#### [Database Migrations - Flyway](references/flyway.md)
- Migration file naming
- Version control patterns
- Schema validation
- Rollback testing

## Directory Structure

```
testing-security-stack/
├── docker-compose.yml      # Main service definitions
├── Makefile                # Automation commands
├── code/                   # Source code to scan
│   └── cai/               # CAI security agents
├── reports/               # All test outputs
│   ├── allure-results/    # Allure report data
│   ├── semgrep/          # Semgrep findings
│   ├── trivy/            # Trivy vulnerabilities
│   ├── zap/              # ZAP reports
│   ├── nuclei/           # Nuclei findings
│   ├── gitleaks/         # Secret detection
│   ├── newman/           # API test results
│   ├── k6/               # Performance metrics
│   ├── locust/           # Load test reports
│   ├── artillery/        # Scenario test reports
│   ├── playwright/       # E2E test results
│   └── cai/              # AI security reports
├── scripts/               # Test scripts
│   ├── run-all-scans.sh
│   └── run-performance-tests.sh
├── zap-config/           # ZAP automation configs
├── newman-collections/   # Postman collections
├── k6-scripts/           # K6 test scripts
├── locust-scripts/       # Locust test files
├── artillery-scripts/    # Artillery configs
├── playwright-tests/     # Playwright tests
└── wiremock/             # Mock service configs
    ├── mappings/
    └── files/
```

## Resources

- **SonarQube**: https://docs.sonarqube.org/
- **Semgrep**: https://semgrep.dev/docs/
- **Trivy**: https://aquasecurity.github.io/trivy/
- **OWASP ZAP**: https://www.zaproxy.org/docs/
- **Nuclei**: https://docs.projectdiscovery.io/tools/nuclei/
- **K6**: https://k6.io/docs/
- **Playwright**: https://playwright.dev/docs/intro
- **Allure**: https://docs.qameta.io/allure/

---

**Remember**: This skill provides comprehensive testing and security stack knowledge. For specific tool details, refer to the reference documentation files.
