# AI Security Reference

Comprehensive guide to AI-powered security testing with CAI (Cybersecurity AI).

## CAI Overview

CAI is an open-source framework for building and deploying AI-powered offensive and defensive security automation. It provides specialized agents for penetration testing, vulnerability discovery, forensics, and security assessment.

**Container**: Custom build from `kalilinux/kali-rolling`
**Type**: AI Security Agent Framework
**Privileged Mode**: Yes (required for security tools)

### Starting CAI

```bash
# Start CAI container
docker-compose up -d cai

# Attach to CAI shell
docker exec -it cai /opt/cai/start.sh

# Or with specific agent
CAI_AGENT_TYPE=bug_bounter_agent docker exec -it cai /opt/cai/start.sh
```

## Available Agents

| Agent | Description | Best Use Case |
|-------|-------------|---------------|
| **redteam_agent** | Offensive security specialist | Active exploitation, vulnerability discovery |
| **blueteam_agent** | Defensive security expert | Security hardening, incident response |
| **bug_bounter_agent** | Bug bounty hunter | Web app security, API testing |
| **dfir_agent** | Digital Forensics and IR | Log analysis, forensic investigation |
| **reverse_engineering_agent** | Binary analysis | Malware analysis, firmware reversing |
| **memory_analysis_agent** | Memory dump analysis | RAM forensics, process analysis |
| **network_security_analyzer_agent** | Network packet analysis | PCAP analysis, traffic inspection |
| **wifi_security_agent** | Wireless security | WiFi pentesting, WPA cracking |
| **android_sast** | Android static analysis | Mobile app security testing |
| **one_tool_agent** | Single-tool execution | Quick scans, specific operations |
| **retester_agent** | Vulnerability retesting | Re-validates discovered vulnerabilities |
| **reporting_agent** | Report generation | Security report creation |

## Agent Capabilities Matrix

| Capability | redteam | blueteam | bug_bounty | dfir | reverse_eng |
|-----------|---------|----------|------------|------|-------------|
| Web App Testing | ⭐⭐⭐ | ⭐ | ⭐⭐⭐⭐⭐ | ⭐⭐ | ⭐ |
| Network Analysis | ⭐⭐⭐⭐ | ⭐⭐⭐⭐⭐ | ⭐⭐ | ⭐⭐⭐ | ⭐ |
| Binary Analysis | ⭐⭐ | ⭐ | ⭐ | ⭐⭐ | ⭐⭐⭐⭐⭐ |
| Forensics | ⭐⭐ | ⭐⭐⭐ | ⭐ | ⭐⭐⭐⭐⭐ | ⭐⭐⭐ |
| Exploit Development | ⭐⭐⭐⭐⭐ | ⭐ | ⭐⭐⭐ | ⭐ | ⭐⭐⭐⭐ |

## Built-in Tools

Tools are organized by security kill chain phases:

### 1. Reconnaissance
- `generic_linux_command` - Execute Linux commands
- `shodan_search` - Shodan API queries
- `google_search` - Google dorking
- `c99_subdomain_finder` - Subdomain discovery
- `nmap` - Network scanning
- `curl` - HTTP requests
- `wget` - File downloads

### 2. Exploitation
- `execute_code` - Code execution
- `ssh_command` - SSH remote commands
- `metasploit` - Metasploit framework integration

### 3. Network Analysis
- `capture_remote_traffic` - Traffic capture
- `netcat` - Network connections
- `netstat` - Network statistics
- `tcpdump` - Packet capture

### 4. Cryptography
- `crypto_tools` - Encryption/decryption utilities

## Workflow Patterns

### Pattern 1: Web Application Pentest

```bash
# 1. Start reconnaissance
CAI>/agent bug_bounter_agent
CAI> Enumerate the target https://target.com

# 2. Discovery phase
CAI> Find all subdomains for target.com using available tools

# 3. Vulnerability scanning
CAI> Scan the application for OWASP Top 10 vulnerabilities

# 4. Manual testing
CAI> Test the login endpoint for SQL injection

# 5. Switch to exploitation
CAI>/agent redteam_agent
CAI> Attempt to exploit the discovered SQL injection vulnerability

# 6. Document findings
CAI>/agent reporting_agent
CAI> Generate a security assessment report for target.com
```

### Pattern 2: Network Security Assessment

```bash
# 1. Network discovery
CAI>/agent redteam_agent
CAI> Perform network discovery on 192.168.1.0/24

# 2. Service enumeration
CAI> Enumerate services on discovered hosts

# 3. Vulnerability assessment
CAI> Check for known vulnerabilities on discovered services

# 4. Traffic analysis
CAI>/agent network_security_analyzer_agent
CAI> Analyze the captured traffic for suspicious activity

# 5. Defense recommendations
CAI>/agent blueteam_agent
CAI> Provide mitigation recommendations for discovered vulnerabilities
```

### Pattern 3: Incident Response

```bash
# 1. Log analysis
CAI>/agent dfir_agent
CAI> Analyze the system logs in /var/log for indicators of compromise

# 2. Memory analysis
CAI>/agent memory_analysis_agent
CAI> Analyze the memory dump for malicious processes

# 3. Network forensics
CAI>/agent network_security_analyzer_agent
CAI> Analyze the PCAP file for command and control traffic

# 4. Malware analysis
CAI>/agent reverse_engineering_agent
CAI> Analyze the suspicious binary found on the system

# 5. Incident report
CAI>/agent reporting_agent
CAI> Create an incident response report with timeline and IOCs
```

### Pattern 4: API Security Testing

```bash
# 1. API discovery
CAI>/agent bug_bounter_agent
CAI> Discover API endpoints from the Swagger documentation at https://api.target.com/docs

# 2. Authentication testing
CAI> Test authentication mechanisms for weaknesses

# 3. Authorization testing
CAI> Test for IDOR vulnerabilities across API endpoints

# 4. Injection testing
CAI> Test API endpoints for injection vulnerabilities

# 5. Rate limiting
CAI> Test rate limiting on sensitive endpoints
```

### Pattern 5: Container Security

```bash
# 1. Image analysis
CAI>/agent blueteam_agent
CAI> Analyze the Docker image for vulnerabilities and misconfigurations

# 2. Runtime security
CAI> Check container runtime configuration for security issues

# 3. Network policies
CAI> Review container network policies and isolation

# 4. Secrets management
CAI> Check for exposed secrets in container environment
```

## Interactive Commands

```bash
# Switch agents during session
CAI>/agent redteam_agent

# List available agents
CAI>/agent list

# Get agent info
CAI>/agent info bug_bounter_agent

# Check API costs
CAI>/cost

# Load previous session
CAI>/load logs/session.jsonl

# Save current session
CAI>/save logs/my-session.jsonl

# Clear conversation
CAI>/clear

# Exit CAI
CAI>/exit
```

## Environment Configuration

Create `/code/cai/dockerized/config/.env`:

```bash
# Required - At least one API key
OPENAI_API_KEY="sk-..."
ANTHROPIC_API_KEY="sk-ant-..."

# Optional AI providers
DEEPSEEK_API_KEY=""
OLLAMA_API_BASE="http://host.docker.internal:11434"

# Optional reconnaissance APIs
C99_API_KEY=""         # For subdomain discovery
SHODAN_API_KEY=""      # For Shodan queries

# Runtime settings
CAI_STREAM=false       # Stream output
CAI_MODEL=gpt-4        # Default model
CAI_AGENT_TYPE=redteam_agent  # Default agent
CAI_MAX_TURNS=50       # Max conversation turns
CAI_PRICE_LIMIT="5.00" # API cost limit

# Display settings
PROMPT_TOOLKIT_NO_CPR=1
```

## Model Support

| Provider | Models | Notes |
|----------|--------|-------|
| OpenAI | GPT-4, GPT-4o, GPT-3.5 | Requires OPENAI_API_KEY |
| Anthropic | Claude 3.5, Claude 3 | Requires ANTHROPIC_API_KEY |
| DeepSeek | DeepSeek-V2 | Requires DEEPSEEK_API_KEY |
| Ollama | Llama, Mistral, Qwen | Local deployment |
| OpenRouter | 100+ models | Via custom base URL |

## Creating Custom Agents

```python
from cai.sdk.agents import Agent, OpenAIChatCompletionsModel
from cai.tools.reconnaissance.generic_linux_command import generic_linux_command
from cai.tools.reconnaissance.nmap import nmap
from openai import AsyncOpenAI

custom_agent = Agent(
    name="Custom Pentester",
    description="Specialized offensive security testing agent",
    instructions="""
    You are a professional penetration tester. Your objectives:
    1. Identify vulnerabilities in target systems
    2. Document all findings with evidence
    3. Provide remediation recommendations
    4. Follow responsible disclosure practices
    5. Stay within authorized scope
    """,
    tools=[
        generic_linux_command,
        nmap,
    ],
    model=OpenAIChatCompletionsModel(
        model="gpt-4",
        openai_client=AsyncOpenAI(),
    ),
)
```

## Creating Custom Tools

```python
from cai.sdk.agents import function_tool

@function_tool
async def check_ssl_certificate(domain: str) -> str:
    """Check SSL certificate validity and configuration.

    Args:
        domain: The domain to check SSL certificate for.
    """
    import ssl
    import socket
    from datetime import datetime

    context = ssl.create_default_context()
    with socket.create_connection((domain, 443)) as sock:
        with context.wrap_socket(sock, server_hostname=domain) as ssock:
            cert = ssock.getpeercert()

    expiry = datetime.strptime(cert['notAfter'], '%b %d %H:%M:%S %Y %Z')
    days_until_expiry = (expiry - datetime.now()).days

    return f"""
    SSL Certificate for {domain}:
    - Issuer: {cert.get('issuer')}
    - Subject: {cert.get('subject')}
    - Valid Until: {cert.get('notAfter')}
    - Days Until Expiry: {days_until_expiry}
    - Version: {cert.get('version')}
    """

@function_tool
async def port_scan(target: str, ports: str = "1-1000") -> str:
    """Perform a port scan on the target.

    Args:
        target: Target IP or hostname.
        ports: Port range to scan (default: 1-1000).
    """
    import subprocess
    result = subprocess.run(
        ['nmap', '-p', ports, target],
        capture_output=True,
        text=True
    )
    return result.stdout
```

## Guardrails

CAI includes built-in protection against prompt injection and unauthorized actions:

```python
from cai.sdk.agents import Agent, InputGuardrail, GuardrailFunctionOutput
from cai.sdk.agents import Runner

async def safety_guardrail(ctx, agent, input_data):
    """Validate input before processing."""
    # Check for dangerous patterns
    dangerous_patterns = [
        'rm -rf /',
        'format c:',
        'drop database',
    ]

    for pattern in dangerous_patterns:
        if pattern.lower() in input_data.lower():
            return GuardrailFunctionOutput(
                output_info="Blocked: Dangerous command detected",
                tripwire_triggered=True,
            )

    return GuardrailFunctionOutput(
        output_info="Input validated",
        tripwire_triggered=False,
    )

agent = Agent(
    name="Safe Security Agent",
    input_guardrails=[
        InputGuardrail(guardrail_function=safety_guardrail),
    ],
)
```

## Output and Reports

Reports are saved to `./reports/cai/`:

```
reports/cai/
├── session_*.jsonl     # Session logs
├── findings/           # Discovered vulnerabilities
│   ├── sqli_001.md
│   ├── xss_001.md
│   └── auth_bypass_001.md
└── reports/            # Generated reports
    ├── pentest_report.md
    └── incident_report.md
```

### Report Format

```markdown
# Security Assessment Report

## Executive Summary
[High-level findings and recommendations]

## Scope
- Target: https://target.com
- Date: 2024-01-15
- Type: Web Application Penetration Test

## Findings Summary
| ID | Title | Severity | Status |
|----|-------|----------|--------|
| VULN-001 | SQL Injection in Login | Critical | Open |
| VULN-002 | XSS in Search | High | Open |

## Detailed Findings

### VULN-001: SQL Injection in Login
**Severity**: Critical
**Location**: POST /api/login
**Description**: The login endpoint is vulnerable to SQL injection...
**Evidence**: [Screenshots/logs]
**Remediation**: Use parameterized queries...

## Recommendations
1. Implement input validation
2. Use prepared statements
3. Deploy WAF

## Appendix
- Tool outputs
- Raw logs
- Technical details
```

## Integration with Stack

- Runs alongside other security tools (ZAP, Nuclei, Semgrep)
- Can analyze code in `./code/` directory
- Reports integrate with Allure dashboard
- Works with WireMock for API testing scenarios

## Best Practices

1. **Match Agent to Task** - Use specialized agents for specific domains
2. **Switch Agents Freely** - Don't hesitate to change agents mid-session
3. **Monitor Costs** - Use `/cost` to track API usage
4. **Save Sessions** - Load successful approaches with `/load`
5. **Use Guardrails** - Enable safety checks for production use
6. **Document Everything** - Record all findings with evidence
7. **Verify Findings** - Use retester_agent to confirm vulnerabilities
8. **Stay in Scope** - Only test authorized targets
9. **Report Responsibly** - Follow disclosure guidelines
10. **Review AI Output** - Validate AI-generated commands before execution
