---
name: api-catalog
description: Search, identify, and suggest APIs from a curated catalog with documentation fetching, implementation code generation, and API comparison capabilities. Use when the user needs API recommendations, wants to integrate with external services, asks about available APIs for a specific task, needs implementation examples for API integration, or wants to compare different API options (REST, GraphQL, gRPC, or third-party SaaS APIs).
---

# API Catalog and Integration Assistant

Access and leverage a curated catalog of APIs to identify, suggest, and implement API integrations across REST, GraphQL, gRPC, and third-party SaaS platforms. **Supports OpenAPI/Swagger specifications for machine-readable API documentation.**

## Core Capabilities

1. **API Discovery** - Search the API catalog by task, category, or service name
2. **OpenAPI Parsing** - Fetch and parse OpenAPI/Swagger specs from URLs
3. **Documentation Access** - Fetch and parse API documentation from source URLs
4. **Code Generation** - Generate implementation code for API integration
5. **API Comparison** - Compare multiple API options for the same use case
6. **Spec Storage** - Store and manage local OpenAPI specs for quick reference

## API Catalog Structure

### Primary Catalog: `references/api-index.md`

Contains curated API entries with:
- API names and descriptions
- **OpenAPI spec URLs** (when available)
- Documentation URLs
- Categories and use cases
- Authentication methods
- Key features

### Local OpenAPI Specs: `references/openapi-specs/`

Store frequently-used OpenAPI specs locally:
```
references/openapi-specs/
├── coverkit.yaml       # CoverKit API spec
├── stripe.yaml         # Stripe API spec
├── twilio.yaml         # Twilio API spec
└── ...
```

---

## OpenAPI/Swagger Support

### Understanding OpenAPI Specs

OpenAPI specifications are machine-readable YAML/JSON files that define:
- **Endpoints** - Available paths and HTTP methods
- **Parameters** - Query, path, header, and body parameters
- **Schemas** - Request/response data structures
- **Authentication** - Security schemes (API key, OAuth2, etc.)
- **Examples** - Sample requests and responses

### Workflow: Parsing OpenAPI from URL

When you have an OpenAPI spec URL (like `https://docs.coverkit.io/openapi.yaml`):

1. **Fetch the spec**
   ```
   Use WebFetch to retrieve the OpenAPI YAML/JSON from the URL
   ```

2. **Parse key sections**
   - `info` - API title, version, description
   - `servers` - Base URLs for the API
   - `paths` - All available endpoints
   - `components/schemas` - Data models
   - `components/securitySchemes` - Auth methods

3. **Extract endpoint details**
   For each path, identify:
   - HTTP method (GET, POST, PUT, DELETE)
   - Path parameters (e.g., `/users/{id}`)
   - Query parameters
   - Request body schema
   - Response schemas and status codes
   - Required authentication

4. **Generate documentation summary**
   Present to user:
   - Overview of the API
   - List of endpoints grouped by tag/category
   - Authentication requirements
   - Key data models

### Workflow: Generating Code from OpenAPI

When generating implementation code from an OpenAPI spec:

1. **Identify the target endpoint**
   - Match user's request to specific path/operation
   - Extract operationId if available

2. **Gather endpoint details**
   ```yaml
   # Example endpoint structure
   /api/v1/policies:
     post:
       operationId: createPolicy
       summary: Create a new insurance policy
       security:
         - bearerAuth: []
       requestBody:
         content:
           application/json:
             schema:
               $ref: '#/components/schemas/PolicyCreate'
       responses:
         '201':
           description: Policy created
           content:
             application/json:
               schema:
                 $ref: '#/components/schemas/Policy'
   ```

3. **Generate typed code**
   - Create interface/type definitions from schemas
   - Build request function with proper parameters
   - Include authentication headers
   - Add response type handling
   - Include error handling

4. **Example output (TypeScript)**
   ```typescript
   interface PolicyCreate {
     policyNumber: string;
     holderName: string;
     coverageType: 'basic' | 'premium' | 'comprehensive';
     startDate: string;
   }

   interface Policy extends PolicyCreate {
     id: string;
     createdAt: string;
     status: 'active' | 'pending' | 'cancelled';
   }

   async function createPolicy(
     data: PolicyCreate,
     token: string
   ): Promise<Policy> {
     const response = await fetch('https://api.coverkit.io/api/v1/policies', {
       method: 'POST',
       headers: {
         'Content-Type': 'application/json',
         'Authorization': `Bearer ${token}`
       },
       body: JSON.stringify(data)
     });

     if (!response.ok) {
       throw new Error(`API error: ${response.status}`);
     }

     return response.json();
   }
   ```

### Workflow: Discovering API from Swagger UI Page

When given a Swagger UI documentation page (like `https://docs.coverkit.io/`):

1. **Identify the OpenAPI spec URL**
   - Look for `url:` in SwaggerUIBundle config
   - Common patterns:
     - `/openapi.yaml`
     - `/openapi.json`
     - `/swagger.json`
     - `/api-docs`

2. **Fetch the raw spec**
   - Construct full URL: `{base_url}{spec_path}`
   - Example: `https://docs.coverkit.io/openapi.yaml`

3. **Parse and catalog**
   - Follow the OpenAPI parsing workflow above

---

## Standard Workflows

### Workflow: Suggesting APIs

When a user asks for API recommendations:

1. **Understand the requirement**
   - Extract the task/use case (e.g., "send SMS", "payment processing")
   - Identify constraints (language, platform, pricing preferences)

2. **Search the catalog**
   - Read `references/api-index.md`
   - Filter APIs by category or keyword match
   - Prioritize APIs with OpenAPI specs available

3. **Present options**
   - Suggest 2-3 most relevant APIs
   - Highlight key differences (pricing, features, complexity)
   - Include documentation links and OpenAPI spec URLs

4. **Provide implementation guidance**
   - If OpenAPI spec available: Parse and generate typed code
   - Reference `references/implementation-patterns.md` for patterns
   - Explain authentication flow

### Workflow: Fetching API Documentation

When detailed API information is needed:

1. **Check for OpenAPI spec first**
   - If spec URL exists in catalog, fetch and parse it
   - This provides structured, complete API information

2. **Fall back to documentation**
   - Use WebFetch to retrieve documentation from URLs
   - Extract key information manually

3. **Summarize findings**
   - Authentication requirements
   - Endpoint patterns
   - Request/response formats
   - Rate limits
   - Common use cases

### Workflow: Comparing APIs

When the user asks to compare APIs:

1. **Gather specs for each API**
   - Fetch OpenAPI specs where available
   - Note which APIs have structured specs vs manual docs

2. **Create comparison matrix**
   | Feature | API A | API B | API C |
   |---------|-------|-------|-------|
   | OpenAPI Spec | Yes | No | Yes |
   | Endpoints | 45 | ~20 | 32 |
   | Auth Method | OAuth2 | API Key | Bearer |
   | Rate Limits | 1000/hr | 100/hr | Unlimited |
   | SDK Support | JS, Python | JS only | All major |

3. **Provide recommendation**
   - Best for specific use cases
   - Trade-offs to consider
   - Ease of integration (OpenAPI = easier)

---

## Adding APIs to the Catalog

### Entry Format with OpenAPI Support

```markdown
## API Name

**Category**: Payment Processing
**Documentation**: https://docs.example.com/
**OpenAPI Spec**: https://docs.example.com/openapi.yaml
**Authentication**: OAuth2 / API Key / Bearer Token

### Description
Brief description of what the API does.

### Key Features
- Feature 1
- Feature 2

### Key Endpoints (from OpenAPI)
- `POST /payments` - Create payment
- `GET /payments/{id}` - Get payment details
- `POST /refunds` - Process refund
```

### Adding New APIs

When the user shares new API sources:

1. **Check for OpenAPI spec**
   - Look for `/openapi.yaml`, `/openapi.json`, `/swagger.json`
   - Check if Swagger UI page reveals spec location

2. **If spec exists**:
   - Fetch and parse the spec
   - Auto-generate catalog entry with endpoints
   - Optionally save spec to `references/openapi-specs/`

3. **If no spec**:
   - Create manual entry from documentation
   - Note that OpenAPI spec is not available

4. **Update catalog**
   - Add to `references/api-index.md`
   - Organize by category

---

## Quick Reference

### OpenAPI Spec Locations (Common Patterns)

| Service | Spec URL Pattern |
|---------|-----------------|
| Standard | `/openapi.yaml` or `/openapi.json` |
| Swagger | `/swagger.json` or `/v2/swagger.json` |
| API Docs | `/api-docs` or `/api-docs.json` |
| Versioned | `/api/v1/openapi.yaml` |

### Authentication Types in OpenAPI

```yaml
# API Key
securitySchemes:
  apiKey:
    type: apiKey
    in: header
    name: X-API-Key

# Bearer Token
securitySchemes:
  bearerAuth:
    type: http
    scheme: bearer

# OAuth2
securitySchemes:
  oauth2:
    type: oauth2
    flows:
      authorizationCode:
        authorizationUrl: https://example.com/oauth/authorize
        tokenUrl: https://example.com/oauth/token
        scopes:
          read: Read access
          write: Write access
```

### API Types

- **REST APIs**: HTTP-based, JSON/XML, standard methods (GET, POST, etc.)
- **GraphQL**: Query language, single endpoint, flexible data fetching
- **gRPC**: Protocol buffers, binary protocol, high performance
- **SaaS APIs**: Third-party services (Stripe, Twilio, AWS, etc.)

---

## Key Principles

1. **OpenAPI first** - Always check for OpenAPI spec before manual parsing
2. **Structured data** - Prefer machine-readable specs for accuracy
3. **Working code** - Generate complete, typed, runnable implementations
4. **Full context** - Include authentication, error handling, types
5. **Keep updated** - Store and version important specs locally
6. **Catalog maintenance** - Keep the index organized with spec URLs
