---
name: sketchnote
description: Generate sketchnote-style visual notes from text content using Google Gemini's Nano Banana Pro image generation API. Use this skill when the user wants to create sketchnotes, visual summaries, meeting notes, or hand-drawn style visual notes from articles, transcripts, or any text content.
---

# Sketchnote Generator Skill

This skill automates the creation of sketchnote-style visual notes using Google Gemini's image generation API (Nano Banana Pro). It transforms text content into authentic hand-drawn style visual notes with doodles, icons, connectors, and hand-lettering.

## Quick Start

```python
from sketchnote_generator import SketchnoteGenerator

# Initialize with API key
generator = SketchnoteGenerator(api_key="your-google-api-key")

# Generate from raw text
result = generator.generate(
    raw_text="Your meeting notes or article content here...",
    layout="dense",  # Layout preset
    aspect_ratio="1:1",  # Square for social
    resolution="2K"
)

print(f"Saved to: {result.output_path}")
```

## What Are Sketchnotes?

Sketchnotes are visual notes that combine:
- **Hand-drawn icons** and doodles
- **Hand-lettered typography** with banners and frames
- **Visual connectors** (arrows, lines, branches)
- **Emphasis elements** (boxes, bubbles, highlights)
- **Dense information** packed into a single visual

This skill captures the authentic Mike Rohde sketchnote aesthetic - like conference notes or visual meeting summaries.

## Workflow

### Step 1: Content Parsing

The skill extracts key structural elements from your text:
- **Title**: Main topic or headline
- **Key Concepts**: Major ideas to visualize
- **Relationships**: How concepts connect
- **Quotes/Callouts**: Important phrases to emphasize

### Step 2: Sketchnote Generation

Prompts are constructed for authentic sketchnote style:
- Dense visual packing with hand-drawn aesthetics
- Multiple interconnected idea clusters
- Banner ribbon headings and thought bubbles
- Mind-map branching structures
- Margin annotations and visual metaphors

## Configuration Options

### Aspect Ratios
- `1:1` - Square (ideal for sketchnotes, Instagram)
- `4:3` - Traditional notebook
- `16:9` - Presentation slides
- `9:16` - Vertical (Stories, mobile)

### Resolutions
- `1K` - Fast generation, smaller files
- `2K` - Balanced quality (default)
- `4K` - Maximum quality (Nano Banana Pro only)

### Layout Presets

| Layout | Description |
|--------|-------------|
| `dense` | Maximum information density, conference notes style |
| `centered` | Central topic with radiating concepts |
| `linear` | Top-to-bottom sequential flow |
| `grid` | Organized grid of concept boxes |
| `mind-map` | Branching tree structure |

### Color Palettes

| Palette | Colors |
|---------|--------|
| `classic` | Marker blue (#1E40AF), gray, yellow highlight |
| `warm` | Brown ink, cream paper, orange accents |
| `cool` | Teal ink, white paper, blue accents |
| `mono` | Black ink only on cream paper |
| `vibrant` | Multiple marker colors, rainbow accents |

## API Methods

### `SketchnoteGenerator(api_key, model=None)`
Initialize the generator.
- `api_key`: Google API key with Gemini access
- `model`: Override default model (optional)

### `parse_content(raw_text) -> ParsedContent`
Extract key concepts and relationships from text.
- Returns: `ParsedContent(title, concepts, relationships, quotes)`

### `generate(raw_text, layout, aspect_ratio, resolution, palette, output_path) -> GenerationResult`
Full generation pipeline.
- Returns: `GenerationResult(success, output_path, metadata, prompt_used)`

## Environment Variables

```bash
GOOGLE_API_KEY=your-api-key           # Required
SKETCHNOTE_OUTPUT_DIR=./output        # Optional, default: current dir
SKETCHNOTE_DEFAULT_LAYOUT=dense       # Optional, default: dense
```

## Examples

### Meeting Notes to Sketchnote

```python
generator = SketchnoteGenerator(api_key=key)

meeting_notes = """
# Q4 Planning Meeting

## Key Goals
- Launch new product by October
- Increase revenue 20%
- Expand to 3 new markets

## Action Items
- Sarah: Finalize pricing
- Mike: Hire 2 engineers
- Lisa: Market research

## Timeline
Phase 1: September
Phase 2: October
Phase 3: November
"""

result = generator.generate(
    raw_text=meeting_notes,
    layout="dense",
    aspect_ratio="1:1",
    resolution="2K",
    palette="classic"
)
```

### Article Summary

```python
result = generator.generate(
    raw_text=article_content,
    layout="centered",
    aspect_ratio="4:3",
    resolution="4K",
    output_path="./article-sketchnote.png"
)
```

### Conference Talk Notes

```python
result = generator.generate(
    raw_text=talk_transcript,
    layout="linear",
    aspect_ratio="16:9",
    resolution="2K",
    palette="vibrant"
)
```

## Visual Elements Generated

The sketchnotes include authentic visual elements:

### Typography
- Bold block letters for main headings (inside banners)
- Hand-lettered section headers with underlines
- Casual handwriting for body content
- ALL CAPS for emphasis moments
- Varied sizes showing clear hierarchy

### Icons & Doodles
- Quick sketch icons representing concepts
- Thought bubbles and speech balloons
- Stars, arrows, and pointers
- Simple pictograms and symbols
- Visual metaphors for abstract ideas

### Connectors & Layout
- Arrows linking related concepts
- Branches for hierarchies
- Numbered lists with custom bullets
- Grouping boxes and frames
- Flow lines showing sequences

### Emphasis Elements
- Banner ribbons for headings
- Highlight boxes for key points
- Underlines and circles
- Exclamation marks and stars
- Margin annotations

## Troubleshooting

### Text not rendering clearly
- Reduce content (aim for 5-7 key concepts)
- Use shorter phrases (3-6 words per element)
- Avoid complex hierarchies
- The model works best with SHORT text - long strings may have errors

### Layout looks crowded
- Switch to `centered` or `grid` layout
- Use larger aspect ratio (16:9)
- Simplify content structure

### Safety blocks
- Remove potentially sensitive terms
- Use more abstract descriptions
- Simplify visual suggestions

## Text Rendering Best Practices (from Google Documentation)

Based on [Google's official guidance](https://ai.google.dev/gemini-api/docs/image-generation):

1. **Put text in quotation marks** - Signals exact spelling priority
2. **Keep text SHORT** - 3-6 words per element renders most accurately
3. **Describe typography descriptively** - Use "clean, bold, sans-serif" not font names
4. **Use Gemini 3 Pro Image** - `gemini-3-pro-image-preview` for best text rendering
5. **Spell check your prompt** - Model reflects any errors
6. **Known limitation** - Complex typography may require regeneration

Sources:
- [Gemini API Image Generation](https://ai.google.dev/gemini-api/docs/image-generation)
- [Google Developers Blog - Prompting Tips](https://developers.googleblog.com/en/how-to-prompt-gemini-2-5-flash-image-generation-for-the-best-results/)
- [Firebase AI Logic - Generate Images](https://firebase.google.com/docs/ai-logic/generate-images-gemini)

## File Locations

- Skill: `~/.claude/skills/sketchnote/SKILL.md`
- Implementation: `~/.claude/skills/sketchnote/sketchnote_generator.py`
