# Security Testing Reference

Comprehensive guide to security testing with SAST, DAST, container scanning, and secret detection tools.

## SAST (Static Application Security Testing)

### Semgrep

Fast, lightweight static analysis for security patterns.

**Container**: `returntocorp/semgrep:latest`

#### Basic Commands

```bash
# Auto-detect language and run all rules
docker exec semgrep semgrep --config auto /src

# Run specific rule pack
docker exec semgrep semgrep --config p/security-audit /src

# Output JSON for CI/CD
docker exec semgrep semgrep --config auto --json --output /reports/semgrep.json /src

# Run with metrics disabled
docker exec semgrep semgrep --config auto --metrics=off /src
```

#### Available Rule Packs

| Pack | Description |
|------|-------------|
| `auto` | Auto-detect rules based on project |
| `p/security-audit` | Comprehensive security rules |
| `p/owasp-top-ten` | OWASP Top 10 coverage |
| `p/ci` | Optimized for CI/CD speed |
| `p/secrets` | Secret detection patterns |
| `p/sql-injection` | SQL injection patterns |
| `p/xss` | Cross-site scripting patterns |

#### Custom Rules

```yaml
# semgrep-rules/custom.yaml
rules:
  - id: hardcoded-password
    pattern: password = "$PASSWORD"
    message: "Hardcoded password detected"
    severity: ERROR
    languages: [python, javascript]

  - id: unsafe-eval
    pattern: eval($X)
    message: "Avoid using eval()"
    severity: WARNING
    languages: [javascript, python]
```

### SonarQube

Continuous code quality and security analysis.

**Container**: `sonarqube:10-community`
**URL**: http://localhost:9000
**Default Credentials**: admin/admin

#### Configuration

```properties
# sonar-project.properties
sonar.projectKey=my-project
sonar.projectName=My Project
sonar.sources=src
sonar.tests=tests
sonar.language=php
sonar.sourceEncoding=UTF-8

# Quality gate thresholds
sonar.qualitygate.wait=true
sonar.qualitygate.timeout=300
```

#### Scanner Commands

```bash
# Run scanner with Docker
docker run --rm \
  -e SONAR_HOST_URL="http://sonarqube:9000" \
  -e SONAR_LOGIN="admin" \
  -e SONAR_PASSWORD="admin" \
  -v "${PWD}:/usr/src" \
  sonarsource/sonar-scanner-cli

# Via sonar-scanner CLI
sonar-scanner \
  -Dsonar.projectKey=my-project \
  -Dsonar.sources=. \
  -Dsonar.host.url=http://localhost:9000 \
  -Dsonar.login=YOUR_TOKEN
```

#### Quality Profiles

| Profile | Languages | Focus |
|---------|-----------|-------|
| Sonar way | All | Default balanced rules |
| Security | All | Security-focused rules |
| OWASP | All | OWASP compliance |

### Trivy

Container and filesystem vulnerability scanner.

**Container**: `aquasec/trivy:latest`

#### Scan Commands

```bash
# Filesystem scan
docker exec trivy trivy fs /src

# Container image scan
docker exec trivy trivy image nginx:latest

# Repository scan
docker exec trivy trivy repo https://github.com/org/repo

# SBOM generation
docker exec trivy trivy fs --format spdx-json -o /reports/sbom.json /src

# Ignore unfixed vulnerabilities
docker exec trivy trivy fs --ignore-unfixed /src
```

#### Severity Filtering

```bash
# High and Critical only
docker exec trivy trivy fs --severity HIGH,CRITICAL /src

# Exit code on findings
docker exec trivy trivy fs --severity CRITICAL --exit-code 1 /src

# Skip specific vulnerabilities
docker exec trivy trivy fs --skip-vuln CVE-2022-1234 /src
```

#### Output Formats

| Format | Flag | Use Case |
|--------|------|----------|
| Table | (default) | Human readable |
| JSON | `--format json` | CI/CD parsing |
| SARIF | `--format sarif` | GitHub integration |
| HTML | `--format html` | Reports |
| SBOM | `--format spdx-json` | Software Bill of Materials |

## DAST (Dynamic Application Security Testing)

### OWASP ZAP

Dynamic application security testing for web applications.

**Container**: `zaproxy/zap-stable:latest`
**API Port**: 8082

#### Scan Types

```bash
# Baseline scan (passive only, fast)
docker exec owasp-zap zap-baseline.py -t http://target:8080 -r /zap/reports/baseline.html

# Full scan (active + passive)
docker exec owasp-zap zap-full-scan.py -t http://target:8080 -r /zap/reports/full.html

# API scan (OpenAPI/Swagger)
docker exec owasp-zap zap-api-scan.py -t http://target:8080/swagger.json -f openapi

# GraphQL scan
docker exec owasp-zap zap-api-scan.py -t http://target:8080/graphql -f graphql
```

#### Automation Framework

```yaml
# zap-config/automation.yaml
env:
  contexts:
    - name: "Default Context"
      urls:
        - "http://target:8080"
      includePaths:
        - "http://target:8080.*"
      excludePaths:
        - ".*logout.*"
        - ".*signout.*"
      authentication:
        method: "form"
        parameters:
          loginUrl: "http://target:8080/login"
          loginRequestData: "username={%username%}&password={%password%}"
        verification:
          method: "response"
          loggedInRegex: "\\Q<a href=\"logout\">"
      users:
        - name: "testuser"
          credentials:
            username: "test@example.com"
            password: "password123"
  parameters:
    failOnError: true
    failOnWarning: false
    progressToStdout: true

jobs:
  - type: spider
    parameters:
      context: "Default Context"
      maxDuration: 5
      maxDepth: 10

  - type: spiderAjax
    parameters:
      context: "Default Context"
      maxDuration: 5

  - type: passiveScan-wait
    parameters:
      maxDuration: 10

  - type: activeScan
    parameters:
      context: "Default Context"
      maxRuleDurationInMins: 5
      maxScanDurationInMins: 30

  - type: report
    parameters:
      template: "traditional-html"
      reportDir: "/zap/reports"
      reportFile: "zap-report"
      reportTitle: "Security Assessment Report"
```

#### API Commands

```bash
# Check ZAP status
curl http://localhost:8082/JSON/core/view/version/

# Start spider
curl "http://localhost:8082/JSON/spider/action/scan/?url=http://target:8080"

# Check spider status
curl "http://localhost:8082/JSON/spider/view/status/?scanId=0"

# Get alerts
curl "http://localhost:8082/JSON/core/view/alerts/"

# Generate report
curl "http://localhost:8082/OTHER/core/other/htmlreport/"
```

### Nuclei

Fast, template-based vulnerability scanner.

**Container**: `projectdiscovery/nuclei:latest`

#### Commands

```bash
# Update templates
docker exec nuclei nuclei -update-templates

# Basic scan
docker exec nuclei nuclei -target http://target:8080 -o /reports/nuclei.txt

# Specific template category
docker exec nuclei nuclei -target http://target:8080 -tags cve -o /reports/cves.txt

# Multiple targets
docker exec nuclei nuclei -list /targets/urls.txt -o /reports/nuclei.txt

# Severity filter
docker exec nuclei nuclei -target http://target:8080 -severity critical,high
```

#### Template Categories

| Category | Flag | Description |
|----------|------|-------------|
| CVE | `-tags cve` | Known CVEs |
| Misconfig | `-tags misconfig` | Misconfigurations |
| Exposure | `-tags exposure` | Information disclosure |
| Takeover | `-tags takeover` | Subdomain takeover |
| Panel | `-tags panel` | Admin panels |
| Tech | `-tags tech` | Technology detection |

## Secret Detection

### Gitleaks

Git repository secret detection.

**Container**: `zricethezav/gitleaks:latest`

#### Commands

```bash
# Detect secrets in directory
docker exec gitleaks gitleaks detect --source /src --report-path /reports/gitleaks.json

# Without git history
docker exec gitleaks gitleaks detect --source /src --no-git

# Verbose output
docker exec gitleaks gitleaks detect --source /src -v

# Exit code on findings
docker exec gitleaks gitleaks detect --source /src --exit-code 1

# Protect mode (pre-commit)
docker exec gitleaks gitleaks protect --staged
```

#### Custom Rules

```toml
# .gitleaks.toml
title = "Custom Gitleaks Rules"

[[rules]]
id = "custom-api-key"
description = "Custom API Key Pattern"
regex = '''(?i)custom[_-]?api[_-]?key[=:]["']?([a-zA-Z0-9]{32,})["']?'''
tags = ["api", "key"]

[[rules]]
id = "internal-password"
description = "Internal Password Pattern"
regex = '''(?i)internal[_-]?password[=:]["']?([^\s"']+)["']?'''
tags = ["password"]

[allowlist]
paths = [
  '''test/.*''',
  '''docs/.*''',
]
```

## Security Testing Patterns

### Pattern 1: Shift-Left Security

```
Developer Commit
    │
    ▼
Pre-commit Hook ───────► Gitleaks (secrets)
    │
    ▼
CI Pipeline Start
    │
    ├─► SAST ─────────────► Semgrep + Trivy
    │
    ├─► Build Container
    │
    ├─► Container Scan ───► Trivy
    │
    ├─► Deploy to Test
    │
    └─► DAST ─────────────► ZAP + Nuclei
    │
    ▼
Quality Gate ◄──────────── Results Aggregation
    │
    ├─► Pass ──► Deploy
    │
    └─► Fail ──► Block + Notify
```

### Pattern 2: Defense in Depth

```bash
#!/bin/bash
# run-security-pipeline.sh

set -e

echo "=== Phase 1: Secret Detection ==="
docker exec gitleaks gitleaks detect --source /src --exit-code 1

echo "=== Phase 2: Static Analysis ==="
docker exec semgrep semgrep --config auto --json /src | tee /reports/semgrep.json
docker exec trivy trivy fs --severity HIGH,CRITICAL --exit-code 0 /src | tee /reports/trivy.txt

echo "=== Phase 3: Container Security ==="
docker exec trivy trivy image --severity CRITICAL --exit-code 1 myapp:latest

echo "=== Phase 4: Dynamic Testing ==="
docker exec owasp-zap zap-baseline.py -t http://target:8080 -J /zap/reports/baseline.json

echo "=== Phase 5: Template-Based Scanning ==="
docker exec nuclei nuclei -target http://target:8080 -severity critical,high -json -o /reports/nuclei.json

echo "All security checks passed!"
```

### Pattern 3: Authenticated Scanning

```yaml
# ZAP with authentication
env:
  contexts:
    - name: "Authenticated Context"
      urls:
        - "http://target:8080"
      authentication:
        method: "json"
        parameters:
          loginUrl: "http://target:8080/api/login"
          loginRequestData: '{"email":"{%username%}","password":"{%password%}"}'
        verification:
          method: "response"
          loggedInRegex: '"token":'
      sessionManagement:
        method: "headers"
        parameters:
          headers:
            - "Authorization: Bearer {%session%}"
```

## Quality Gates

### SAST Quality Gate

```yaml
# Example CI configuration
sast:
  rules:
    critical:
      max: 0
      action: block
    high:
      max: 0
      action: block
    medium:
      max: 10
      action: warn
    low:
      max: 50
      action: report
```

### DAST Quality Gate

```yaml
dast:
  rules:
    # ZAP alert thresholds
    risk_high:
      max: 0
      action: block
    risk_medium:
      max: 5
      action: warn
    risk_low:
      action: report
```

## Integration Examples

### GitHub Actions

```yaml
name: Security Scan
on: [push, pull_request]

jobs:
  security:
    runs-on: ubuntu-latest
    steps:
      - uses: actions/checkout@v3

      - name: Semgrep SAST
        uses: returntocorp/semgrep-action@v1
        with:
          config: p/security-audit

      - name: Trivy Scan
        uses: aquasecurity/trivy-action@master
        with:
          scan-type: 'fs'
          severity: 'CRITICAL,HIGH'
          exit-code: '1'

      - name: Gitleaks
        uses: gitleaks/gitleaks-action@v2
```

### GitLab CI

```yaml
include:
  - template: Security/SAST.gitlab-ci.yml
  - template: Security/Secret-Detection.gitlab-ci.yml
  - template: Security/Dependency-Scanning.gitlab-ci.yml

semgrep-sast:
  stage: test
  image: returntocorp/semgrep
  script:
    - semgrep --config auto --json --output semgrep.json .
  artifacts:
    reports:
      sast: semgrep.json
```

## Best Practices

1. **Run SAST before DAST** - Static analysis is faster and catches issues earlier
2. **Use multiple tools** - Different tools find different issues
3. **Customize rules** - Tune for your technology stack
4. **Handle false positives** - Create allowlists for known safe patterns
5. **Aggregate results** - Centralize findings for triage
6. **Set quality gates** - Block deployments on critical findings
7. **Scan dependencies** - Include SCA in your pipeline
8. **Test authentication** - Configure authenticated scanning
9. **Include API testing** - Scan API endpoints specifically
10. **Monitor trends** - Track security posture over time
